package com.bxm.warcar.canal.mq;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.warcar.canal.CanalEntity;
import com.bxm.warcar.canal.CanalEventListener;
import com.bxm.warcar.canal.CanalEventType;
import com.bxm.warcar.mq.ConsumeStatus;
import com.bxm.warcar.mq.Message;
import com.bxm.warcar.mq.SingleMessageListener;
import com.google.common.collect.Maps;
import org.apache.commons.lang.ArrayUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;

/**
 * @author allen
 * @since 1.0.0
 */
public class CanalMessageListenerDispatcher implements SingleMessageListener {

    private final static Logger LOGGER = LoggerFactory.getLogger(CanalMessageListenerDispatcher.class);

    private final String topic;
    private final Map<String, CanalEventListener<? extends CanalEntity>> listeners = Maps.newHashMap();

    public CanalMessageListenerDispatcher(String topic, CanalEventListener<? extends CanalEntity>... listeners) {
        this.topic = topic;
        if (ArrayUtils.isNotEmpty(listeners)) {
            for (CanalEventListener<? extends CanalEntity> listener : listeners) {
                this.listeners.put(listener.listening(), listener);
            }
        }
    }

    @Override
    public String getTopic() {
        return topic;
    }

    @Override
    public ConsumeStatus consume(Message message, Object context) {
        byte[] body = message.getBody();
        if (ArrayUtils.isEmpty(body)) {
            if (LOGGER.isWarnEnabled()) {
                LOGGER.warn("[{}] message body has be null.", message.getMsgId());
            }
            return ConsumeStatus.CONSUME_SUCCESS;
        }

        try {
            CanalMessageBody entity = JSONObject.parseObject(body, CanalMessageBody.class);

            String tags = message.getTags();
            CanalEventListener<? extends CanalEntity> listener = listeners.get(tags);
            doConsume(entity, listener);

            return ConsumeStatus.CONSUME_SUCCESS;
        } catch (Exception e) {
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("[" + message.getMsgId() + "] message consume fail.", e);
            }
            return ConsumeStatus.RECONSUME_LATER;
        }
    }

    private <T extends CanalEntity> void doConsume(CanalMessageBody entity, CanalEventListener<T> listener) {
        if (null == listener) {
            return;
        }
        CanalEventType canalEventType = entity.getCanalEventType();
        T after = forceConvert(entity.getAfter(), listener);
        T before = forceConvert(entity.getBefore(), listener);
        if (canalEventType == CanalEventType.INSERT) {
            listener.doInsert(before);
        } else if (canalEventType == CanalEventType.DELETE) {
            listener.doDelete(before);
        } else if (canalEventType == CanalEventType.UPDATE) {
            listener.doUpdate(before, after);
        }
    }

    private  <T extends CanalEntity> T forceConvert(Object obj, CanalEventListener<T> listener) {
        if (null != obj && obj instanceof JSON) {
            return JSONObject.toJavaObject((JSON) obj, listener.getEntityClass());
        }
        return null;
    }

    public Map<String, CanalEventListener<? extends CanalEntity>> getListeners() {
        return listeners;
    }
}
