package com.bxm.warcar.canal;

import com.alibaba.otter.canal.client.CanalConnector;
import com.alibaba.otter.canal.client.CanalConnectors;
import com.alibaba.otter.canal.protocol.CanalEntry;
import com.alibaba.otter.canal.protocol.Message;
import com.alibaba.otter.canal.protocol.exception.CanalClientException;
import com.bxm.warcar.canal.utils.EntityHelper;
import com.bxm.warcar.utils.LifeCycle;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.Map;

/**
 * @author allen
 * @since 1.0.0
 */
public class CanalClient extends LifeCycle {

    private static final Logger LOGGER = LoggerFactory.getLogger(CanalClient.class);

    public static final String FILTER_DEFAULT = ".*\\..*";
    public static final int BATCH_SIZE = 1000;

    private final Map<String, CanalEventListener<? extends CanalEntity>> listeners = Maps.newConcurrentMap();

    private final String zkServers;
    private final String destination;
    private final String username;
    private final String password;
    private String filter;
    private int batchSize = BATCH_SIZE;

    private CanalConnector connector;


    public CanalClient(String zkServers, String destination) {
        this(zkServers, destination, null, null);
    }

    public CanalClient(String zkServers, String destination, String username, String password) {
        this.zkServers = zkServers;
        this.destination = destination;
        this.username = username;
        this.password = password;
    }

    @Override
    protected void doInit() {
        connector = CanalConnectors.newClusterConnector(zkServers, destination, username, password);
        try {
            connector.connect();
            connector.subscribe(filter);
            connector.rollback();

            while (true) {
                Message message = connector.getWithoutAck(batchSize);
                long messageId = message.getId();
                List<CanalEntry.Entry> entries = message.getEntries();
                int size = entries.size();
                boolean empty = messageId == -1 || size == 0;
                if (!empty) {
                    try {
                        this.process(entries);
                        connector.ack(messageId);
                    } catch (Exception e) {
                        if (LOGGER.isErrorEnabled()) {
                            LOGGER.error("process:", e);
                        }
                        connector.rollback(messageId);
                    }
                }

                try {
                    Thread.sleep(1000);
                } catch (InterruptedException e) {
                    if (LOGGER.isErrorEnabled()) {
                        LOGGER.error("sleep:", e);
                    }
                }
            }

        } catch (CanalClientException e) {
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("doInit:", e);
            }
        }
    }

    @Override
    protected void doDestroy() {
        connector.disconnect();
    }

    private void process(List<CanalEntry.Entry> entries) {
        if (CollectionUtils.isEmpty(entries)) {
            return;
        }

        for (CanalEntry.Entry entry : entries) {
            CanalEntry.EntryType entryType = entry.getEntryType();
            if (entryType == CanalEntry.EntryType.TRANSACTIONBEGIN || entryType == CanalEntry.EntryType.TRANSACTIONEND) {
                continue;
            }

            CanalEntry.RowChange rowChage = null;
            try {
                rowChage = CanalEntry.RowChange.parseFrom(entry.getStoreValue());
            } catch (Exception e) {
                throw new RuntimeException("ERROR ## parser of eromanga-event has an error , data:" + entry.toString(), e);
            }

            CanalEntry.EventType eventType = rowChage.getEventType();
            String schemaName = entry.getHeader().getSchemaName();
            String tableName = entry.getHeader().getTableName();

            String listening = schemaName + "." + tableName;
            CanalEventListener<? extends CanalEntity> listener = listeners.get(listening);

            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug(String.format("Process message: binlog[%s:%s] , name[%s,%s] , eventType : %s",
                        entry.getHeader().getLogfileName(), entry.getHeader().getLogfileOffset(),
                        schemaName, tableName,
                        eventType));
            }

            if (null == listener) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("No canal event listener of {}.", listening);
                }
                continue;
            }

            for (CanalEntry.RowData rowData : rowChage.getRowDatasList()) {
                if (eventType == CanalEntry.EventType.DELETE) {
                    doDelete(rowData.getBeforeColumnsList(), listener);
                } else if (eventType == CanalEntry.EventType.INSERT) {
                    doInsert(rowData.getAfterColumnsList(), listener);
                } else {
                    doUpdate(rowData.getBeforeColumnsList(), rowData.getAfterColumnsList(), listener);
                }
            }
        }
    }

    protected  <T extends CanalEntity> void doDelete(List<CanalEntry.Column> columns, CanalEventListener<T> listener) {
        T o = convert(columns, listener);
        listener.doDelete(o);
    }

    protected <T extends CanalEntity> void doInsert(List<CanalEntry.Column> columns, CanalEventListener<T> listener) {
        T o = convert(columns, listener);
        listener.doInsert(o);
    }

    protected <T extends CanalEntity> void doUpdate(List<CanalEntry.Column> before, List<CanalEntry.Column> after, CanalEventListener<T> listener) {
        T beforeEntity = convert(before, listener);
        T afterEntity = convert(after, listener);
        listener.doUpdate(beforeEntity, afterEntity);
    }

    protected  <T extends CanalEntity> T convert(List<CanalEntry.Column> columns, CanalEventListener<T> listener) {
        return EntityHelper.consutract(columns, listener.getEntityClass());
    }

    public void addListener(CanalEventListener<? extends CanalEntity> listener) {
        if (null == listener) {
            return;
        }
        String listening = listener.listening();
        if (StringUtils.isBlank(listening)) {
            throw new IllegalArgumentException("listening cannot be blank");
        }
        if (listeners.containsKey(listening)) {
            if (LOGGER.isWarnEnabled()) {
                LOGGER.warn("Repeated listener [{}] already existed.", listening);
            }
        }
        listeners.put(listening, listener);
    }

    public void addListeners(Map<String, CanalEventListener<? extends CanalEntity>> listeners) {
        if (null == listeners) {
            return;
        }
        this.listeners.putAll(listeners);
    }

    public String getZkServers() {
        return zkServers;
    }

    public String getDestination() {
        return destination;
    }

    public String getUsername() {
        return username;
    }

    public String getPassword() {
        return password;
    }

    public String getFilter() {
        return filter;
    }

    public void setFilter(String filter) {
        this.filter = filter;
    }

    public int getBatchSize() {
        return batchSize;
    }

    public void setBatchSize(int batchSize) {
        this.batchSize = batchSize;
    }
}
