package com.bxm.warcar.dpl2.plugin.spring;

import com.bxm.warcar.dpl2.PluginRuntimeException;
import com.bxm.warcar.dpl2.hotswap.ClassFilter;
import com.bxm.warcar.dpl2.hotswap.PluginClassLoader;
import com.bxm.warcar.dpl2.hotswap.ResourceFileter;
import com.bxm.warcar.dpl2.plugin.Plugin;
import com.bxm.warcar.dpl2.plugin.PluginConfig;
import com.bxm.warcar.dpl2.plugin.PluginConfigAware;
import com.bxm.warcar.dpl2.plugin.PluginLoader;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.AnnotationConfigApplicationContext;
import org.springframework.context.support.AbstractApplicationContext;
import org.springframework.core.io.ClassPathResource;

import java.io.IOException;
import java.lang.annotation.Annotation;
import java.util.List;
import java.util.Map;
import java.util.Properties;

/**
 * @author allen
 * @date 2019/6/27
 * @since 1.0.0
 */
@Slf4j
public class SpringPluginLoader implements PluginLoader {

    private static final List<Class<? extends Annotation>> DEFAULT_LOADING_ANNOTATION = Lists.newArrayList(
            org.springframework.stereotype.Component.class,
            org.springframework.stereotype.Repository.class,
            org.springframework.stereotype.Service.class,
            org.springframework.stereotype.Controller.class
    );

    private final String basePackage;
    private final List<Class<? extends Annotation>> loadAnnotations;
    private final ApplicationContext parent;

    public SpringPluginLoader(String basePackage, ApplicationContext parent) {
        this(basePackage, DEFAULT_LOADING_ANNOTATION, parent);
    }

    public SpringPluginLoader(String basePackage, List<Class<? extends Annotation>> loadAnnotations, ApplicationContext parent) {
        this.basePackage = basePackage;
        this.loadAnnotations = loadAnnotations;
        this.parent = parent;
    }

    @Override
    public Plugin load(PluginConfig config) {
        AbstractApplicationContext applicationContext = register(config);
        return new SpringPlugin(config, applicationContext);
    }

    private AbstractApplicationContext register(PluginConfig config) {
        ClassLoader currentClassLoader = Thread.currentThread().getContextClassLoader();
        try {
            PluginClassLoader pluginClassLoader = new PluginClassLoader(config.getJarPath());
            Thread.currentThread().setContextClassLoader(pluginClassLoader);

            AnnotationConfigApplicationContext applicationContext = new AnnotationConfigApplicationContext();

            List<Class<?>> classesNeeded = pluginClassLoader.searchClasses(basePackage, new ClassFilter() {
                @Override
                public boolean accept(Class<?> clazz) {
                    if (CollectionUtils.isNotEmpty(loadAnnotations)) {
                        for (Class<? extends Annotation> loadClass : loadAnnotations) {
                            if (clazz.isAnnotationPresent(loadClass)) {
                                return true;
                            }
                        }
                    }
                    return false;
                }
            });
            if (CollectionUtils.isEmpty(classesNeeded)) {
                log.warn("Plugin[{}] No annotations to load. {}", config.getJarPath(), loadAnnotations);
                return null;
            }
            this.fillConfig(config, pluginClassLoader);

            applicationContext.setParent(parent);
            applicationContext.setClassLoader(pluginClassLoader);
            applicationContext.register(classesNeeded.toArray(new Class<?>[0]));
            applicationContext.refresh();

            Map<String, PluginConfigAware> beans = applicationContext.getBeansOfType(PluginConfigAware.class);
            for (PluginConfigAware aware : beans.values()) {
                aware.setPluginConfig(config);
            }

            return applicationContext;
        } finally {
            Thread.currentThread().setContextClassLoader(currentClassLoader);
        }
    }

    private void fillConfig(PluginConfig config, PluginClassLoader pluginClassLoader) {
        List<String> resources = pluginClassLoader.searchResources("META-INF/*", new ResourceFileter() {
            @Override
            public boolean accept(String name) {
                return StringUtils.endsWith(name, "plugin.properties");
            }
        }, false);
        if (CollectionUtils.isEmpty(resources)) {
            throw new PluginRuntimeException("Plugin[" + config.getJarPath() + "] Can not found file classpath:/META-INF/plugin.properties");
        }
        for (String resource : resources) {
            try {
                Properties properties = new Properties();
                properties.load(new ClassPathResource(resource).getInputStream());
                config.setProperties(properties);
            } catch (IOException e) {
                log.error("Can not load[{}] resource for class path.", resource);
            }
        }
    }
}
