package com.bxm.warcar.integration.dc.dot;

import com.google.common.base.Preconditions;
import org.apache.commons.lang.StringUtils;

/**
 * DotParamter factory
 *
 * @author allen
 * @date 2019/3/7
 * @since 1.0.0
 */
public class DotParameterFactory {

    private VerEnum verEnum;
    private DotParameter parameters = new DotParameter();

    private DotParameterFactory() {}

    /**
     * create dot paramter factory of version 2 for product ads
     */
    public static DotParameterFactory create() {
        return create(VerEnum.V2);
    }

    /**
     * create dot paramter factory for product ads
     * @param ver dot service version
     */
    public static DotParameterFactory create(VerEnum ver) {
        return create("ads", ver);
    }

    /**
     * create dot paramter factory
     * @param product product, e.g. "ads"
     * @param ver dot service version
     */
    public static DotParameterFactory create(String product, VerEnum ver) {
        Preconditions.checkArgument(StringUtils.isNotBlank(product));
        Preconditions.checkNotNull(ver);
        DotParameterFactory factory = new DotParameterFactory().p(product);
        factory.verEnum = ver;
        return factory;
    }

    public DotParameterFactory p(String p) {
        parameters.put("p", p);
        return this;
    }

    public DotParameterFactory ip(String ip) {
        parameters.put("ip", ip);
        return this;
    }

    public DotParameterFactory ua(String userAgent) {
        parameters.put("user_agent", userAgent);
        return this;
    }

    public DotParameterFactory referrer(String referrer) {
        parameters.put("referrer", referrer);
        return this;
    }

    /**
     * 屏幕分辨率的高：window.screen.height
     * @param sh px
     */
    public DotParameterFactory sh(Integer sh) {
        parameters.put("sh", sh);
        return this;
    }

    /**
     * 屏幕分辨率的宽：window.screen.width
     * @param sw px
     */
    public DotParameterFactory sw(Integer sw) {
        parameters.put("sw", sw);
        return this;
    }

    /**
     * model type
     * @param mt value
     * @see ModelTypeEnum
     * @see #mt(int)
     */
    public DotParameterFactory mt(ModelTypeEnum mt) {
        if (verEnum == VerEnum.V1) {
            parameters.put("mt", mt.getDotV1());
        } else if (verEnum == VerEnum.V2) {
            parameters.put("mt", mt.getDotV2());
        }
        return this;
    }

    /**
     * 自由设置 modelType
     * @param modelType modelType
     */
    public DotParameterFactory mt(int modelType) {
        parameters.put("mt", modelType);
        return this;
    }

    /**
     * bxmid
     * @param bxmid value
     */
    public DotParameterFactory bxmid(String bxmid) {
        parameters.put("bxm_id", bxmid);
        return this;
    }

    public DotParameterFactory x(Integer x) {
        parameters.put("x", x);
        return this;
    }

    public DotParameterFactory y(Integer y) {
        parameters.put("y", y);
        return this;
    }

    public DotParameterFactory popupId(Object popupId) {
        parameters.put("popupId", popupId);
        return this;
    }

    /**
     * 广告券ID，老参数 preid
     * @param awardid value
     */
    public DotParameterFactory awardId(Object awardid) {
        parameters.put("awardid", awardid);
        return this;
    }

    public DotParameterFactory assetsId(Object assetsId) {
        parameters.put("assetsId", assetsId);
        return this;
    }

    /**
     * 弹窗类型
     */
    public DotParameterFactory scene(String scene) {
        parameters.put("scene", scene);
        return this;
    }

    /**
     * 时间戳
     */
    public DotParameterFactory ts(Long ts) {
        parameters.put("ts", ts);
        return this;
    }

    public DotParameterFactory price(Long price) {
        parameters.put("price", price);
        return this;
    }

    /**
     * 券来源
     * @param stype 直投-1，非直投-0，默认为0
     */
    public DotParameterFactory stype(Integer stype) {
        parameters.put("s_type", stype);
        return this;
    }

    public DotParameterFactory appkey(String appkey) {
        parameters.put("appkey", appkey);
        return this;
    }

    /**
     * money-前缀
     * @param business value
     */
    public DotParameterFactory business(String business) {
        parameters.put("business", business);
        return this;
    }

    public DotParameterFactory activityId(Object activityId) {
        parameters.put("activityid", activityId);
        return this;
    }

    public DotParameterFactory uid(String uid) {
        parameters.put("uid", uid);
        return this;
    }

    public DotParameterFactory spm(String spm) {
        parameters.put("spm", spm);
        return this;
    }

    public DotParameterFactory imei(String imei) {
        parameters.put("i", imei);
        return this;
    }

    public DotParameterFactory idfa(String idfa) {
        parameters.put("f", idfa);
        return this;
    }

    /**
     * 扩展字段，格式ext=s:1,x:2
     * @param ext value
     */
    public DotParameterFactory ext(String ext) {
        parameters.put("ext", ext);
        return this;
    }

    /**
     * 客户端地址栏信息
     * @param locaurl value
     */
    public DotParameterFactory locaurl(String locaurl) {
        parameters.put("locaurl", locaurl);
        return this;
    }

    /**
     * @param modelType value
     */
    public DotParameterFactory modelType(String modelType) {
        parameters.put("modeltype", modelType);
        return this;
    }

    /**
     * @param userAgent value
     */
    public DotParameterFactory userAgent(String userAgent) {
        parameters.put("user_agent", userAgent);
        return this;
    }

    /**
     * @param status value
     */
    public DotParameterFactory status(String status) {
        parameters.put("status", status);
        return this;
    }

    /**
     * @param toType value
     */
    public DotParameterFactory toType(String toType) {
        parameters.put("toType", toType);
        return this;
    }

    public DotParameter build() {
        return this.parameters;
    }
}
