package com.bxm.warcar.integration.distributed;

import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.KeyBuilder;

import java.time.Duration;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.function.Predicate;

/**
 * 首次执行器
 * <p>
 * 在给定的静默时间内，确保一个操作只被执行一次。这在分布式环境中非常有用，可以防止在短时间内重复处理相同的任务。
 * 例如，当多个服务实例同时收到相同的事件时，只有一个实例能够成功执行关联的操作。
 *
 * @param <T> 执行器接受的参数类型
 * @author Allen Hu
 * @date 2025/10/15
 */
public class FirstTimeExecutor<T> {

    private final Counter counter;
    private final String executorName;
    private final Duration silentTime;
    private final Consumer<T> executor;

    /**
     * 构造一个新的 FirstTimeExecutor
     *
     * @param counter      分布式计数器，用于跟踪执行次数
     * @param executorName 执行器的唯一名称，用于生成分布式锁的键
     * @param silentTime   静默时间。在此期间，操作最多只执行一次
     */
    public FirstTimeExecutor(Counter counter, String executorName, Duration silentTime) {
        this(counter, executorName, silentTime, null);
    }

    /**
     * 构造一个新的 FirstTimeExecutor
     *
     * @param counter      分布式计数器，用于跟踪执行次数
     * @param executorName 执行器的唯一名称，用于生成分布式锁的键
     * @param silentTime   静默时间。在此期间，操作最多只执行一次
     * @param executor     要执行的操作
     */
    public FirstTimeExecutor(Counter counter, String executorName, Duration silentTime, Consumer<T> executor) {
        this.counter = counter;
        this.executorName = executorName;
        this.silentTime = silentTime;
        this.executor = executor;
    }

    /**
     * 尝试执行操作
     * <p>
     * 只有在静默时间窗口内的第一次调用，操作才会被实际执行。
     *
     * @param t 执行器要消费的数据
     */
    public void submit(T t) {
        execute(t, this.executor);
    }

    /**
     * 尝试执行操作
     * <p>
     * 只有在静默时间窗口内的第一次调用，操作才会被实际执行。
     *
     * @param t        执行器要消费的数据
     * @param executor 要执行的操作
     */
    public void execute(T t, Consumer<T> executor) {
        Objects.requireNonNull(executor);
        KeyGenerator key = () -> KeyBuilder.build("warcar", "condition_executor", executorName);
        int expireTimeInSecond = (int) silentTime.getSeconds();
        Long after = counter.incrementAndGet(key, expireTimeInSecond, new Predicate<Long>() {
            @Override
            public boolean test(Long after) {
                // 只有当计数器值从0变为1时，才返回true
                return after == 1;
            }
        });
        if (after == 1) {
            executor.accept(t);
        }
    }
}
