package com.bxm.warcar.integration.pair.local;

import com.bxm.warcar.integration.pair.DefaultValue;
import com.bxm.warcar.integration.pair.Pair;
import com.bxm.warcar.integration.pair.Value;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

/**
 * 基于本地缓存的实现
 *
 * @author allen
 * @since 1.0.0
 */
public class LoadingCachePairImpl implements Pair {

    private static final Logger LOGGER = LoggerFactory.getLogger(LoadingCachePairImpl.class);
    private final LoadingCache<String, Optional<String>> cache;
    private final Pair source;

    /**
     * @param souce 数据来源
     */
    public LoadingCachePairImpl(Pair souce) {
        this(souce, 5, TimeUnit.MINUTES);
    }

    /**
     * @param source 来源
     * @param duration 刷新间隔时间
     * @param timeUnit 刷新间隔时间单位
     */
    public LoadingCachePairImpl(Pair source, int duration, TimeUnit timeUnit) {
        this.cache = CacheBuilder.newBuilder()
                .refreshAfterWrite(duration, timeUnit)
                .expireAfterWrite(duration * 2, timeUnit)
                .build(new CacheLoader<String, Optional<String>>() {
                    @Override
                    public Optional<String> load(String key) throws Exception {
                        Value value = source.get(key);
                        return Optional.ofNullable(value.of());
                    }
                });
        this.source = source;
    }

    @Override
    public Value get(String key) {
        return new DefaultValue(get0(key));
    }

    @Override
    public Collection<String> keys() {
        // 有可能出现 源 已更新，但是本地缓存未更新，最后无法获取到实时 Value 的情况。
        return source.keys();
    }

    private String get0(String key) {
        try {
            Optional<String> optional = this.cache.get(key);
            return optional.orElse(null);
        } catch (ExecutionException e) {
            LOGGER.error("get: ", e);
            return null;
        }
    }
}
