package com.bxm.warcar.integration.pushable.message;

import com.bxm.warcar.cache.push.Pushable;
import com.bxm.warcar.integration.message.AbstractMessageListener;
import com.bxm.warcar.integration.pushable.CachePushableBus;
import com.bxm.warcar.mq.Message;
import com.bxm.warcar.utils.StringHelper;
import com.google.common.base.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;

/**
 * 基于消息监听器的缓存推送消费者
 * @author allen
 * @since 1.0.0
 */
public class CachePushableMessageListener extends AbstractMessageListener {

    private static final Logger LOGGER = LoggerFactory.getLogger(CachePushableMessageListener.class);

    private final String topic;
    private final String consumerId;
    private final CachePushableBus cachePushableBus;
    private final boolean ignoreNonePushable;

    public CachePushableMessageListener(String topic, String consumerId, CachePushableBus cachePushableBus) {
        this(topic, consumerId, cachePushableBus, false);
    }

    public CachePushableMessageListener(String topic, String consumerId, CachePushableBus cachePushableBus, boolean ignoreNonePushable) {
        Preconditions.checkNotNull(cachePushableBus);
        this.topic = topic;
        this.consumerId = consumerId;
        this.cachePushableBus = cachePushableBus;
        this.ignoreNonePushable = ignoreNonePushable;
    }

    @Override
    public String getTopic() {
        return topic;
    }

    @Override
    public String getConsumerId() {
        return consumerId;
    }

    protected MessageEntity convert(Object request) {
        return null;
    }

    @Override
    protected void consume(Message message, Object request, Object returning) {
        MessageEntity entity = null;
        if (!(request instanceof MessageEntity)) {
            if (LOGGER.isWarnEnabled()) {
                LOGGER.warn("request object must be instance of com.bxm.warcar.integration.pushable.message.MessageEntity!");
            }
            entity = convert(request);
            if (null == entity) {
                if (LOGGER.isInfoEnabled()) {
                    LOGGER.info("You can override #convert(Object request)!");
                }
                return;
            }
        } else {
            entity = (MessageEntity) request;
        }
        String serviceName = entity.getServiceName();
        Map<String, Object> parameters = entity.getParameters();
        byte[] data = entity.getData();
        Pushable bean = cachePushableBus.getRefreshable(serviceName);
        if (null == bean) {
            if (ignoreNonePushable) {
                return;
            }
            throw new NullPointerException("Cannot found Pushable of serviceName: " + serviceName);
        }

        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Starting refresh cache, serviceName={}, parameters={}, data={}",
                    serviceName, parameters, StringHelper.convert(data));
        }
        long start = System.currentTimeMillis();
        bean.push(parameters, data);
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Cache refreshed in {} ms. serviceName={}, parameters={}, data-length={}",
                    (System.currentTimeMillis() - start), serviceName, parameters, (null == data ? 0 : data.length));
        }
    }
}