package com.bxm.warcar.integration.distributed.leader;

import com.bxm.warcar.zk.ZkClientHolder;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.curator.framework.recipes.leader.LeaderLatch;
import org.apache.curator.framework.recipes.leader.LeaderLatchListener;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;

import java.io.Closeable;
import java.io.IOException;
import java.util.List;

/**
 * <p>LeaderLatchExecutorBeanFactory</p>
 * <p>
 * 用于自动为实现 LeaderLatchExecutor 接口的 Spring Bean 创建并管理 LeaderLatch 实例，实现分布式 Leader 选举。
 * 在 Bean 初始化后，自动注册 LeaderLatch，并在成为 Leader 时调用对应 Bean 的 execute 方法。
 * 实现 Closeable 接口，容器销毁时自动关闭所有 LeaderLatch，释放资源。
 * </p>
 * 主要功能：<br/>
 * 1. 自动发现并处理 LeaderLatchExecutor 类型的 Bean。<br/>
 * 2. 为每个 Bean 创建 LeaderLatch，并监听 Leader 状态变化。<br/>
 * 3. 成为 Leader 时执行 Bean 的业务逻辑。<br/>
 * 4. 统一管理 LeaderLatch 生命周期，支持自动关闭。<br/>
 *
 * @author Allen Hu
 * @date 2025/10/20
 */
@Slf4j
public class LeaderLatchExecutorBeanFactory implements BeanPostProcessor, Closeable {

    private final ZkClientHolder zkClientHolder;
    private final List<LeaderLatch> leaderLatchList = Lists.newArrayList();

    public LeaderLatchExecutorBeanFactory(ZkClientHolder zkClientHolder) {
        this.zkClientHolder = zkClientHolder;
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        Class<?> targetClass = AopUtils.getTargetClass(bean);

        boolean assignableFrom = LeaderLatchExecutor.class.isAssignableFrom(targetClass);
        if (assignableFrom) {
            LeaderLatchExecutor leaderLatchExecutor = (LeaderLatchExecutor) bean;
            this.createLeaderLatchAndStart(leaderLatchExecutor);
        }

        return bean;
    }

    @Override
    public void close() {
        leaderLatchList.forEach((e) -> {
            try {
                e.close();
            } catch (IOException ex) {
                log.error("LeaderLatch close error", ex);
            }
        });
    }

    private void createLeaderLatchAndStart(LeaderLatchExecutor leaderLatchExecutor) {
        String className = leaderLatchExecutor.getClass().getName();
        LeaderLatch leaderLatch = new LeaderLatch(zkClientHolder.get(), "/warcar/leader-latch/" + className);
        leaderLatch.addListener(new LeaderLatchListener() {
            @Override
            public void isLeader() {
                try {
                    leaderLatchExecutor.execute();
                } catch (Exception e) {
                    log.error("LeaderLatchExecutor execute error", e);
                }
            }

            @Override
            public void notLeader() {
            }
        });
        leaderLatchList.add(leaderLatch);
        try {
            leaderLatch.start();
            log.info("LeaderLatch {} start success", className);
        } catch (Exception e) {
            log.error("LeaderLatch start error", e);
        }
    }

}
