package com.bxm.warcar.ip.impl;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.sec.client.FastIPGeoClient;
import com.alibaba.sec.domain.FastGeoConf;
import com.aliyuncs.DefaultAcsClient;
import com.aliyuncs.IAcsClient;
import com.aliyuncs.exceptions.ClientException;
import com.aliyuncs.geoip.model.v20200101.DescribeGeoipInstanceDataInfosRequest;
import com.aliyuncs.geoip.model.v20200101.DescribeGeoipInstanceDataInfosResponse;
import com.aliyuncs.geoip.model.v20200101.DescribeGeoipInstanceDataUrlRequest;
import com.aliyuncs.geoip.model.v20200101.DescribeGeoipInstanceDataUrlResponse;
import com.aliyuncs.profile.DefaultProfile;
import com.bxm.warcar.ip.IP;
import com.bxm.warcar.ip.IpLibrary;
import com.bxm.warcar.ip.impl.aliyun.FileCache;
import com.bxm.warcar.ip.impl.aliyun.GeoipProfile;
import com.bxm.warcar.ip.impl.aliyun.Summary;
import com.bxm.warcar.ip.impl.aliyun.Type;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.LifeCycle;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.google.common.collect.Maps;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import java.io.*;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * @author allen
 * @date 2021-04-20
 * @since 1.0
 */
public class AliyunIpLibrary extends LifeCycle implements IpLibrary {

    private static final Logger LOGGER = LoggerFactory.getLogger(AliyunIpLibrary.class);

    private final ScheduledExecutorService scheduled = new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("upgrade"));

    private final String parentPath = System.getProperty("user.home") + File.separator + ".warcar" + File.separator;
    private final File summaryFile = new File(parentPath + "geoip.summary.json");

    private final GeoipProfile profile;
    private final IAcsClient client;

    private Summary summary = new Summary();
    private FastIPGeoClient fastIPGeoClient;

    public AliyunIpLibrary(GeoipProfile profile) {
        this.profile = profile;
        this.client = new DefaultAcsClient(DefaultProfile.getProfile(profile.getRegionId(), profile.getAccessKeyId(), profile.getSecret()));
    }

    @Override
    public int getOrder() {
        return 0;
    }

    @Override
    protected void doInit() {
        this.mkdirParentIfNecessary(this.parentPath);
        this.downloadGeoipIfNecessary();
        this.refresh();

        this.scheduled.scheduleWithFixedDelay(() -> {
            if (this.downloadGeoipIfNecessary()) {
                this.refresh();
            }
        }, profile.getUpgradeInHours(), profile.getUpgradeInHours(), TimeUnit.HOURS);
    }

    @Override
    protected void doDestroy() {
    }

    @Override
    public IP find(String ip) {
        try {
            String search = this.fastIPGeoClient.search(ip);
            if (StringUtils.isNotBlank(search)) {
                Result result = JsonHelper.convert(search, Result.class);
                IP i = new IP();
                i.setCountry(result.getCountry());
                i.setCountrycode(result.getCountry_code());
                i.setProvince(result.getProvince());
                i.setCity(result.getCity());
                i.setCounty(result.getCounty());
                i.setIsp(result.getIsp());
                i.setIspcode(result.getIsp_code());
                i.setLongitude(result.getLongitude());
                i.setLatitude(result.getLatitude());
                i.setHitcode(findFirst(result.getCounty_code(), result.getCity_code(), result.getProvince_code()));
                return i;
            }
        } catch (Exception e) {
            LOGGER.warn("findIp: {}", ip);
        }
        return null;
    }

    @Override
    public synchronized void refresh() {
        try {
            long start = System.currentTimeMillis();
            FastGeoConf geoConf = new FastGeoConf();
            geoConf.setDataFilePath(this.summary.getProfile().get(Type.IPV4_DATA_TRACE).getPath());
            geoConf.setLicenseFilePath(this.summary.getProfile().get(Type.LICENSE).getPath());

            HashSet<String> set = new HashSet<>(Arrays.asList(
                    "country", "country_code", "province", "province_code",
                    "city", "city_code", "county", "county_code",
                    "isp", "isp_code", "routes",
                    "longitude", "latitude"
            ));
            geoConf.setProperties(set);
            geoConf.filterEmptyValue();
            this.fastIPGeoClient = new FastIPGeoClient(geoConf);
            if (LOGGER.isInfoEnabled()) {
                LOGGER.info("Ip library (aliyun) load finished in {} ms", (System.currentTimeMillis() - start));
            }
        } catch (Exception e) {
            throw new RuntimeException("refresh: ", e);
        }
    }

    private String findFirst(String...array) {
        for (String s : array) {
            if (StringUtils.isNotBlank(s) && !"null".equalsIgnoreCase(s)) {
                return s;
            }
        }
        return null;
    }

    /**
     * 下载地理位置库，如果需要的话
     */
    private boolean downloadGeoipIfNecessary() {
        // 读取上一次已下载的版本简要
        try {
            String json = summaryFile.exists() ? this.readFile(summaryFile) : null;
            if (StringUtils.isNotBlank(json)) {
                this.summary = JsonHelper.convert(json, Summary.class);
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Aliyun Geoip summary: {}", JSONObject.toJSONString(this.summary));
        }

        // 比较版本是否有更新
        try {
            boolean updated = false;
            Map<Type, DescribeGeoipInstanceDataInfosResponse.DataInfo> dataInfos = getDataInfos();
            for (Map.Entry<Type, DescribeGeoipInstanceDataInfosResponse.DataInfo> dataInfo : dataInfos.entrySet()) {
                Type key = dataInfo.getKey();
                String remoteVersion = dataInfo.getValue().getVersion();
                Map<Type, FileCache> profile = this.summary.getProfile();
                FileCache fileCache = profile.get(key);
                String currentVersion = Objects.isNull(fileCache) ? null : fileCache.getVersion();
                if (!StringUtils.equals(currentVersion, remoteVersion)) {
                    // Need refresh
                    LOGGER.info("[{}] discovering different versions, current is {}, remote is {}", key, currentVersion, remoteVersion);
                    DescribeGeoipInstanceDataUrlResponse dataUrlResponse = getUrl(key);
                    String downloadUrl = dataUrlResponse.getFixedDomainDownloadUrl();
                    byte[] bytes = this.downloadFile(downloadUrl);
                    String fileName = StringUtils.defaultIfBlank(getFileNameOnDownloadUrl(downloadUrl), key.name() + "_" + remoteVersion);
                    File file = new File(parentPath + fileName);
                    this.checkAndCreateFile(file);
                    this.writeDataToFile(file, bytes);
                    profile.put(key, new FileCache(file.getPath(), remoteVersion));
                    updated = true;
                }
            }
            // 刷新简要
            if (updated) {
                this.writeDataToFile(summaryFile, JsonHelper.convert2bytes(this.summary));
            }
            return updated;
        } catch (ClientException | IOException e) {
            throw new RuntimeException(e);
        }
    }

    private Map<Type, DescribeGeoipInstanceDataInfosResponse.DataInfo> getDataInfos() throws ClientException {
        DescribeGeoipInstanceDataInfosRequest request = new DescribeGeoipInstanceDataInfosRequest();
        request.setInstanceId(this.profile.getInstanceId());
        request.putQueryParameter("LocationDataType", "TRACE");
        DescribeGeoipInstanceDataInfosResponse response = client.getAcsResponse(request);
        List<DescribeGeoipInstanceDataInfosResponse.DataInfo> dataInfos = response.getDataInfos();
        Map<Type, DescribeGeoipInstanceDataInfosResponse.DataInfo> map = Maps.newHashMap();
        for (DescribeGeoipInstanceDataInfosResponse.DataInfo dataInfo : dataInfos) {
            for (Type value : Type.values()) {
                if (StringUtils.equalsIgnoreCase(value.name(), dataInfo.getType())) {
                    map.put(value, dataInfo);
                    break;
                }
            }
        }
        return map;
    }

    private DescribeGeoipInstanceDataUrlResponse getUrl(Type type) throws ClientException {
        DescribeGeoipInstanceDataUrlRequest dr = new DescribeGeoipInstanceDataUrlRequest();
        dr.setInstanceId(this.profile.getInstanceId());
        dr.setDataType(type.name());
        return client.getAcsResponse(dr);
    }

    private String getFileNameOnDownloadUrl(String downloadUrl) {
        UriComponents build = UriComponentsBuilder.fromUriString(downloadUrl).build();
        List<String> pathSegments = build.getPathSegments();
        return pathSegments.get(pathSegments.size() - 1);
    }

    private void mkdirParentIfNecessary(String parent) {
        File directory = new File(parent);
        if (!directory.exists()) {
            boolean mkdirs = directory.mkdirs();
            if (!mkdirs) {
                throw new RuntimeException(parent + " cannot run command: mkdirs!");
            }
        } else {
            if (!directory.isDirectory()) {
                throw new RuntimeException(parent + " is not directory!");
            }
        }
    }

    private void checkAndCreateFile(File file) {
        if (!file.exists()) {
            try {
                if (!file.createNewFile()) {
                    throw new RuntimeException("Cannot create file: " + file);
                }
                if (LOGGER.isInfoEnabled()) {
                    LOGGER.info("Created file: {}", file);
                }
            } catch (IOException e) {
                throw new RuntimeException("createNewFile: ", e);
            }
        }
        if (!file.isFile()) {
            throw new RuntimeException("File must be not directory: " + file);
        }
    }

    private String readFile(File file) throws IOException {
        return FileUtils.readFileToString(file, StandardCharsets.UTF_8);
    }

    private void writeDataToFile(File file, byte[] data) throws IOException {
        FileUtils.writeByteArrayToFile(file, data);
        if (LOGGER.isInfoEnabled()) {
            LOGGER.info("Data {} bytes write to file {} successful.", data.length, file.getPath());
        }
    }

    private byte[] downloadFile(String url) {
        HttpURLConnection urlConnection = null;
        InputStream inputStream = null;
        ByteArrayOutputStream os = null;
        try {
            urlConnection = (HttpURLConnection) new URL(url).openConnection();
            inputStream = urlConnection.getInputStream();

            int length = urlConnection.getHeaderFieldInt("Content-Length", -1);
            int available = length > 0 ? length : inputStream.available();
            long start = System.currentTimeMillis();
            if (LOGGER.isInfoEnabled()) {
                LOGGER.info("{} Starting download, The content length is {}...", url, available);
            }

            os = new ByteArrayOutputStream(available);

            byte[] buffer = new byte[50 * 1024 * 1024];
            int len;
            while ((len = inputStream.read(buffer)) != -1) {
                os.write(buffer, 0, len);
            }
            os.flush();

            if (LOGGER.isInfoEnabled()) {
                LOGGER.info("{} Finished in {} ms!", url, (System.currentTimeMillis() - start));
            }

            return os.toByteArray();
        } catch (IOException e) {
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("getLicense: ", e);
            }
            return null;
        } finally {
            IOUtils.closeQuietly(inputStream);
            if (null != urlConnection) {
                urlConnection.disconnect();
            }
        }
    }

    private static class Result implements Serializable {

        private static final long serialVersionUID = -4731354983202603658L;
        private String country;
        private String country_code;
        private String province;
        private String province_code;
        private String city;
        private String city_code;
        private String county;
        private String county_code;
        private String isp;
        private String isp_code;
        private String routes;
        private String longitude;
        private String latitude;

        public String getCountry() {
            return country;
        }

        public void setCountry(String country) {
            this.country = country;
        }

        public String getCountry_code() {
            return country_code;
        }

        public void setCountry_code(String country_code) {
            this.country_code = country_code;
        }

        public String getProvince() {
            return province;
        }

        public void setProvince(String province) {
            this.province = province;
        }

        public String getProvince_code() {
            return province_code;
        }

        public void setProvince_code(String province_code) {
            this.province_code = province_code;
        }

        public String getCity() {
            return city;
        }

        public void setCity(String city) {
            this.city = city;
        }

        public String getCity_code() {
            return city_code;
        }

        public void setCity_code(String city_code) {
            this.city_code = city_code;
        }

        public String getCounty() {
            return county;
        }

        public void setCounty(String county) {
            this.county = county;
        }

        public String getCounty_code() {
            return county_code;
        }

        public void setCounty_code(String county_code) {
            this.county_code = county_code;
        }

        public String getIsp() {
            return isp;
        }

        public void setIsp(String isp) {
            this.isp = isp;
        }

        public String getIsp_code() {
            return isp_code;
        }

        public void setIsp_code(String isp_code) {
            this.isp_code = isp_code;
        }

        public String getRoutes() {
            return routes;
        }

        public void setRoutes(String routes) {
            this.routes = routes;
        }

        public String getLongitude() {
            return longitude;
        }

        public void setLongitude(String longitude) {
            this.longitude = longitude;
        }

        public String getLatitude() {
            return latitude;
        }

        public void setLatitude(String latitude) {
            this.latitude = latitude;
        }
    }
}
