package com.bxm.warcar.logging;

import org.apache.commons.lang.StringUtils;

import javax.servlet.http.HttpServletRequest;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

/**
 * @author allen
 * @since 2020-03-18
 */
public class Logging {

    private String applicationId;
    private String requestedSessionId;
    private String scheme;
    private String host;
    private int port;
    private String method;
    private String path;
    private String queryString;
    private Map<String, String[]> params;
    private String clientIp;
    private String serverIp;
    private Map<String, String> headers;
    private Date startTime = new Date();
    private Date endTime;
    private String requestBody;
    private String responseBody;
    private long consumeTimeInMillis;
    private boolean success;
    private String throwable;
    private String operator;

    private Logging() {}

    static Logging build(String applicationId, HttpServletRequest request, long startTimeInMillis,
                         String requestBody, String responseBody, boolean success,
                         String throwable, String operator) {
        Logging o = new Logging();
        o.applicationId = applicationId;
        o.requestedSessionId = request.getRequestedSessionId();
        o.scheme = request.getScheme();
        o.host = request.getServerName();
        o.port = request.getServerPort();
        o.method = request.getMethod();
        o.path = request.getRequestURI();
        o.queryString = request.getQueryString();
        o.params = request.getParameterMap();
        o.clientIp = getIpFromHeader(request);
        o.serverIp = request.getLocalAddr();
        o.headers = getHeaderMap(request);
        o.startTime = new Date(startTimeInMillis);
        o.endTime = new Date();
        o.requestBody = requestBody;
        o.responseBody = responseBody;
        o.consumeTimeInMillis = o.endTime.getTime() - o.startTime.getTime();
        o.success = success;
        o.throwable = throwable;
        o.operator = operator;
        return o;
    }

    private static Map<String, String> getHeaderMap(HttpServletRequest req) {
        Map<String, String> map = new HashMap<>();
        Enumeration enu = req.getHeaderNames();
        while (enu.hasMoreElements()) {
            String headerName = (String) enu.nextElement();
            String headerValue = req.getHeader(headerName);
            map.put(headerName, headerValue);
        }
        return map;
    }

    private static String getIpFromHeader(HttpServletRequest request) {
        if (null == request) {
            return null;
        }
        // TODO 这种情况客户端可直接改写Header，从而到达伪造的目的。这里需要配合Nginx、SLB或者网关来做特殊处理。
        // 比如当SLB->Server环境时，那么需要取 X-Forwarded-For 的最后一个IP，因为第一个可能是伪造的。
        String ip = request.getHeader("X-Forwarded-For");
        if (StringUtils.isNotBlank(ip) && !"unknown".equalsIgnoreCase(ip)) {
            return ip;
        }
        if (org.apache.commons.lang3.StringUtils.isBlank(ip) || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("X-Real-IP");
        }
        if (org.apache.commons.lang3.StringUtils.isBlank(ip) || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("Proxy-Client-IP");
        }
        if (org.apache.commons.lang3.StringUtils.isBlank(ip) || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("WL-Proxy-Client-IP");
        }
        if (org.apache.commons.lang3.StringUtils.isBlank(ip) || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getRemoteAddr();
        }
        return ip;
    }

    public String getApplicationId() {
        return applicationId;
    }

    public String getRequestedSessionId() {
        return requestedSessionId;
    }

    public String getScheme() {
        return scheme;
    }

    public String getHost() {
        return host;
    }

    public int getPort() {
        return port;
    }

    public String getMethod() {
        return method;
    }

    public String getPath() {
        return path;
    }

    public String getQueryString() {
        return queryString;
    }

    public Map<String, String[]> getParams() {
        return params;
    }

    public String getClientIp() {
        return clientIp;
    }

    public String getServerIp() {
        return serverIp;
    }

    public Map<String, String> getHeaders() {
        return headers;
    }

    public Date getStartTime() {
        return startTime;
    }

    public Date getEndTime() {
        return endTime;
    }

    public String getRequestBody() {
        return requestBody;
    }

    public String getResponseBody() {
        return responseBody;
    }

    public long getConsumeTimeInMillis() {
        return consumeTimeInMillis;
    }

    public boolean isSuccess() {
        return success;
    }

    public String getThrowable() {
        return throwable;
    }

    public String getOperator() {
        return operator;
    }

    public void setOperator(String operator) {
        this.operator = operator;
    }

    @Override
    public String toString() {
        return "Logging{" +
                "requestedSessionId='" + requestedSessionId + '\'' +
                ", applicationId='" + applicationId + '\'' +
                ", scheme='" + scheme + '\'' +
                ", host='" + host + '\'' +
                ", port=" + port +
                ", method='" + method + '\'' +
                ", path='" + path + '\'' +
                ", queryString='" + queryString + '\'' +
                ", params=" + params +
                ", clientIp='" + clientIp + '\'' +
                ", serverIp='" + serverIp + '\'' +
                ", headers=" + headers +
                ", startTime=" + startTime +
                ", endTime=" + endTime +
                ", requestBody='" + requestBody + '\'' +
                ", responseBody='" + responseBody + '\'' +
                ", consumeTimeInMillis=" + consumeTimeInMillis +
                ", success=" + success +
                ", throwable='" + throwable + '\'' +
                ", operator='" + operator + '\'' +
                '}';
    }
}
