package com.bxm.warcar.logging;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.web.filter.OncePerRequestFilter;
import org.springframework.web.util.ContentCachingRequestWrapper;
import org.springframework.web.util.ContentCachingResponseWrapper;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.List;

/**
 * @author allen
 * @since 2020-03-18
 */
public class OncePerRequestLoggingFilter extends OncePerRequestFilter {

    private static final int DEFAULT_PAYLOAD_MAX_LENGTH = 1024;

    private final String applicationId;
    private final int payloadMaxLength;
    private final LoggingWriter loggingWriter;
    private final OperatorExtractor operatorExtractor;

    private List<String> excludeRequestUriPrefix;

    public OncePerRequestLoggingFilter(String applicationId, OperatorExtractor operatorExtractor) {
        this(applicationId, DEFAULT_PAYLOAD_MAX_LENGTH, new LoggingWriter() {}, operatorExtractor);
    }

    public OncePerRequestLoggingFilter(String applicationId, int payloadMaxLength, LoggingWriter loggingWriter, OperatorExtractor operatorExtractor) {
        this.applicationId = applicationId;
        this.payloadMaxLength = payloadMaxLength;
        this.loggingWriter = loggingWriter;
        this.operatorExtractor = operatorExtractor;
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
        String requestUri = request.getRequestURI();
        if (CollectionUtils.isNotEmpty(excludeRequestUriPrefix)
                && excludeRequestUriPrefix.stream().anyMatch((e) -> StringUtils.startsWithIgnoreCase(requestUri, e))) {
            filterChain.doFilter(request, response);
            return;
        }

        long startTime = System.currentTimeMillis();

        ContentCachingRequestWrapper requestWrapper = new ContentCachingRequestWrapper(request);
        ContentCachingResponseWrapper responseWrapper = new ContentCachingResponseWrapper(response);

        try {
            filterChain.doFilter(requestWrapper, responseWrapper);
            doLog(startTime, requestWrapper, responseWrapper, null, true);
        } catch (Exception e) {
            doLog(startTime, requestWrapper, responseWrapper, e.getMessage(), false);
            throw e;
        } finally {
            // 如果没有这一步客户端将无法收到响应内容
            responseWrapper.copyBodyToResponse();
        }
    }

    private void doLog(long startTime, ContentCachingRequestWrapper requestWrapper, ContentCachingResponseWrapper responseWrapper,
                       String throwable, boolean success) {
        String requestPayload = convert2String(requestWrapper.getContentAsByteArray(), requestWrapper.getCharacterEncoding());
        String responsePayload = convert2String(responseWrapper.getContentAsByteArray(), responseWrapper.getCharacterEncoding());
        if (null != loggingWriter) {
            String operator = extractOperator(requestWrapper);
            loggingWriter.write(Logging.build(applicationId, requestWrapper, startTime, requestPayload, responsePayload, success, throwable, operator));
        }
    }

    private String extractOperator(HttpServletRequest request) {
        if (null != operatorExtractor) {
            try {
                return operatorExtractor.extract(request);
            } catch (Exception e) {
                // no-op
            }
        }
        return null;
    }

    private String convert2String(byte[] buffer, String characterEncoding) {
        if (ArrayUtils.isEmpty(buffer)) {
            return null;
        }
        int len = buffer.length;
        if (len > 0) {
            int maxLength = Math.min(len, getPayloadMaxLength());
            try {
                return new String(buffer, 0, maxLength, characterEncoding);
            } catch (UnsupportedEncodingException ex) {
                return "[unknown data packet length is " + len + "]";
            }
        }
        return null;
    }

    public int getPayloadMaxLength() {
        return payloadMaxLength;
    }

    public LoggingWriter getLoggingWriter() {
        return loggingWriter;
    }

    public List<String> getExcludeRequestUriPrefix() {
        return excludeRequestUriPrefix;
    }

    public void setExcludeRequestUriPrefix(List<String> excludeRequestUriPrefix) {
        this.excludeRequestUriPrefix = excludeRequestUriPrefix;
    }
}
