package com.bxm.warcar.mq.redis;

import com.bxm.warcar.mq.*;
import com.bxm.warcar.utils.LifeCycle;
import com.bxm.warcar.utils.NamedThreadFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.redis.core.RedisTemplate;

import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * @author allen
 * @date 2019/10/17
 * @since 1.0.0
 */
public class RedisConsumer extends LifeCycle implements Consumer {

    private static final Logger LOGGER = LoggerFactory.getLogger(RedisConsumer.class);
    private final ThreadPoolExecutor consumer = new ThreadPoolExecutor(1, 1, 0, TimeUnit.SECONDS,
            new LinkedBlockingQueue<>(), new NamedThreadFactory("RedisConsumer"));
    private final RedisTemplate<String, Message> redisTemplate;
    private final SingleMessageListener messageListener;
    private final AtomicBoolean shutdown = new AtomicBoolean(false);

    public RedisConsumer(RedisTemplate<String, Message> redisTemplate, SingleMessageListener messageListener) {
        this.redisTemplate = redisTemplate;
        this.messageListener = messageListener;
    }

    @Override
    protected void doInit() {
        String topic = messageListener.getTopic();
        consumer.execute(() -> {
            for (;;) {
                if (shutdown.get()) {
                    break;
                }
                try {
                    Message message = redisTemplate.opsForList().leftPop(RedisConst.key(topic), 1, TimeUnit.SECONDS);
                    if (null != message) {
                        ConsumeStatus status = messageListener.consume(message, redisTemplate);
                        if (status == ConsumeStatus.RECONSUME_LATER) {
                            redisTemplate.opsForList().rightPush(RedisConst.key(topic), message);
                            try {
                                Thread.sleep(500);
                            } catch (InterruptedException ignored) {
                            }
                        }
                    }
                } catch (Exception e) {
                    LOGGER.error("", e);
                }
            }
        });
    }

    @Override
    protected void doDestroy() {
        consumer.shutdown();
        shutdown.compareAndSet(false, true);
        try {
            consumer.awaitTermination(5, TimeUnit.SECONDS);
        } catch (InterruptedException ignore) {
        }
    }

    @Override
    public void suspend() {

    }

    @Override
    public void shutdown() {
        this.doDestroy();
    }

    @Override
    public void start() {
        this.doInit();
    }

    @Override
    public boolean isStarted() {
        return false;
    }

    @Override
    public Listener getMessageListener() {
        return messageListener;
    }
}
