package com.bxm.warcar.sentinel;

import com.alibaba.csp.sentinel.slots.block.authority.AuthorityRule;
import com.alibaba.csp.sentinel.slots.block.authority.AuthorityRuleManager;
import com.alibaba.csp.sentinel.slots.block.degrade.DegradeRule;
import com.alibaba.csp.sentinel.slots.block.degrade.DegradeRuleManager;
import com.alibaba.csp.sentinel.slots.block.flow.FlowRule;
import com.alibaba.csp.sentinel.slots.block.flow.FlowRuleManager;
import com.alibaba.csp.sentinel.slots.system.SystemRule;
import com.alibaba.csp.sentinel.slots.system.SystemRuleManager;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.LifeCycle;
import com.bxm.warcar.zk.ZkClientHolder;
import com.google.common.base.Preconditions;
import org.apache.commons.lang3.StringUtils;
import org.springframework.core.env.ConfigurableEnvironment;

import java.util.List;

/**
 * @author allen
 * @date 2022-03-29
 * @since 1.0
 */
public class ZooKeeperDataSourceRegistryFactory extends LifeCycle {

    private ZooKeeperDataSource<List<FlowRule>> flowRuleDataSource;
    private ZooKeeperDataSource<List<SystemRule>> systemRuleDataSource;
    private ZooKeeperDataSource<List<DegradeRule>> degradeRuleDataSource;
    private ZooKeeperDataSource<List<AuthorityRule>> authorityRuleDataSource;

    public ZooKeeperDataSourceRegistryFactory(ConfigurableEnvironment environment, ZkClientHolder zkClientHolder) {
        String groupId = "sentinel";
        String dataId = getApplicationId(environment);
        flowRuleDataSource = new ZooKeeperDataSource<>(zkClientHolder, groupId, dataId, source -> parse(source).getFlowRules());
        systemRuleDataSource = new ZooKeeperDataSource<>(zkClientHolder, groupId, dataId, source -> parse(source).getSystemRules());
        degradeRuleDataSource = new ZooKeeperDataSource<>(zkClientHolder, groupId, dataId, source -> parse(source).getDegradeRules());
        authorityRuleDataSource = new ZooKeeperDataSource<>(zkClientHolder, groupId, dataId, source -> parse(source).getAuthorityRules());
    }

    private SentinelRuleProperties parse(byte[] json) {
        return JsonHelper.convert(json, SentinelRuleProperties.class);
    }

    private String getApplicationId(ConfigurableEnvironment environment) {
        String name = environment.getProperty("spring.application.name");
        Preconditions.checkArgument(StringUtils.isNotBlank(name), "Not set value of 'spring.application.name'!");
        return name;
    }

    @Override
    protected void doInit() {
        FlowRuleManager.register2Property(flowRuleDataSource.getProperty());
        SystemRuleManager.register2Property(systemRuleDataSource.getProperty());
        DegradeRuleManager.register2Property(degradeRuleDataSource.getProperty());
        AuthorityRuleManager.register2Property(authorityRuleDataSource.getProperty());
    }

    @Override
    protected void doDestroy() {
        try {
            flowRuleDataSource.close();
            systemRuleDataSource.close();
            degradeRuleDataSource.close();
            authorityRuleDataSource.close();
        } catch (Exception ignored) {
        }
    }
}
