package com.bxm.warcar.utils.file;

import com.google.common.base.Preconditions;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

/**
 * @author allen
 * @since 1.0.0
 */
public class SystemFileComparator implements FileComparator {

    private static final Logger LOGGER = LoggerFactory.getLogger(SystemFileComparator.class);

    private final File fingerprint;
    private final File data;
    private final FingerprintTracker fingerprintTracker;
    private final DataTracker dataTracker;

    SystemFileComparator(File fingerprint, File data, FingerprintTracker fingerprintTracker, DataTracker dataTracker) {
        Preconditions.checkNotNull(fingerprintTracker);
        Preconditions.checkNotNull(dataTracker);

        String tmpdir = System.getProperty("java.io.tmpdir");
        if (null == fingerprint) {
            fingerprint = new File(tmpdir + File.separator + "ipipnet.fingerprint");
        }
        if (null == data) {
            data = new File(tmpdir + File.separator + "ipipnet.data");
        }

        checkAndCreateFile(fingerprint);
        checkAndCreateFile(data);
        this.fingerprint = fingerprint;
        this.data = data;
        this.fingerprintTracker = fingerprintTracker;
        this.dataTracker = dataTracker;
    }

    @Override
    public boolean compare() throws IOException {
        Preconditions.checkNotNull(fingerprintTracker);

        String remoteFingerprint = fingerprintTracker.getFingerprint();
        if (StringUtils.isBlank(remoteFingerprint)) {
            throw new RuntimeException("fingerprint cannot be blank!");
        }

        String localFingerprint = readLocalFingerprint();

        if (StringUtils.equals(remoteFingerprint, localFingerprint)) {
            // 指纹一致
            return true;
        }

        Preconditions.checkNotNull(dataTracker);

        byte[] data = dataTracker.getData();

        writeDataFile(data);
        writeLocalFingerprint(remoteFingerprint);
        return false;
    }

    @Override
    public File getData() {
        return data;
    }

    @Override
    public File getFingerprint() {
        return fingerprint;
    }

    private void checkAndCreateFile(File file) {
        if (!file.exists()) {
            try {
                if (!file.createNewFile()) {
                    throw new RuntimeException("Cannot create file: " + file);
                }
                if (LOGGER.isInfoEnabled()) {
                    LOGGER.info("Created file: {}", file);
                }
            } catch (IOException e) {
                throw new RuntimeException("createNewFile: ", e);
            }
        }
        if (!file.isFile()) {
            throw new RuntimeException("File must be not directory: " + file);
        }
    }

    private void writeLocalFingerprint(String fingerprintStr) throws IOException {
        FileUtils.writeStringToFile(fingerprint, fingerprintStr, "utf-8");
    }

    private String readLocalFingerprint() {
        try {
            return FileUtils.readFileToString(fingerprint, "utf-8");
        } catch (IOException e) {
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("read: ", e);
            }
            return null;
        }
    }

    private void writeDataFile(byte[] data) throws IOException {
        FileUtils.writeByteArrayToFile(this.data, data);
    }

    public static void main(String[] args) {
        try {
            String md5 = DigestUtils.md5Hex(new FileInputStream("D:\\workspace\\bxm\\warcar-parent\\warcar-boot\\spring-boot-starter-warcar-ip\\src\\main\\resources\\ipipnet_20180814.datx"));
            System.out.println(md5);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }
}
