package com.bxm.warcar.zk.listener.children;

import com.bxm.warcar.utils.LifeCycle;
import com.bxm.warcar.zk.ZkClientHolder;
import com.bxm.warcar.zk.listener.Changed;
import com.bxm.warcar.zk.listener.ChildrenChangedListener;
import com.bxm.warcar.zk.listener.NodeChangedListener;
import org.apache.curator.framework.recipes.cache.ChildData;
import org.apache.curator.framework.recipes.cache.PathChildrenCache;
import org.apache.curator.framework.recipes.cache.PathChildrenCacheEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;

/**
 * @author allen
 * @since 1.0.0
 */
public class ChildrenChanged extends LifeCycle implements Changed {

    private static final Logger LOGGER = LoggerFactory.getLogger(ChildrenChanged.class);

    private final ZkClientHolder zkClientHolder;
    private final String path;
    private final ChildrenChangedListener listener;

    private PathChildrenCache cache;

    public ChildrenChanged(ZkClientHolder zkClientHolder, String path, ChildrenChangedListener listener) {
        this.zkClientHolder = zkClientHolder;
        this.path = path;
        this.listener = listener;
    }

    @Override
    public void addListener(NodeChangedListener listener) {
        if (null == listener) {
            return;
        }

        if (null != cache) {
            cache.getListenable().addListener((curatorFramework, event) -> {
                ChildData childData = event.getData();
                if (null == childData) {
                    return;
                }
                String path = childData.getPath();
                byte[] data = childData.getData();

                PathChildrenCacheEvent.Type type = event.getType();

                if (listener instanceof ChildrenChangedListener) {
                    ChildrenChangedListener childrenChangedListener = (ChildrenChangedListener) listener;
                    if (type == PathChildrenCacheEvent.Type.CHILD_ADDED) {
                        childrenChangedListener.add(path, data);
                    } else if (type == PathChildrenCacheEvent.Type.CHILD_UPDATED) {
                        childrenChangedListener.update(path ,data);
                    } else if (type == PathChildrenCacheEvent.Type.CHILD_REMOVED) {
                        childrenChangedListener.remove(path, data);
                    }
                } else {
                    if (type == PathChildrenCacheEvent.Type.CHILD_UPDATED) {
                        listener.update(path, data);
                    }
                }
            });
        }
    }

    @Override
    protected void doInit() {
        cache = new PathChildrenCache(zkClientHolder.get(), path, true);
        try {
            cache.start();
        } catch (Exception e) {
            throw new RuntimeException("start: ", e);
        }
        this.addListener(listener);
        if (LOGGER.isInfoEnabled()) {
            LOGGER.info("ZooKeeper path: {} was listening children!", path);
        }
    }

    @Override
    protected void doDestroy() {
        if (null != cache) {
            try {
                cache.close();
            } catch (IOException e) {
                if (LOGGER.isErrorEnabled()) {
                    LOGGER.error("close:", e);
                }
            }
        }
    }
}
