/*
 * Decompiled with CFR 0.152.
 */
package com.sap.db.jdbc;

import com.sap.db.annotations.NotThreadSafe;
import com.sap.db.jdbc.ConnectionProperties;
import com.sap.db.jdbc.ConnectionProperty;
import com.sap.db.jdbc.ConnectionSapDB;
import com.sap.db.jdbc.ParseInfo;
import com.sap.db.jdbc.trace.Tracer;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

@NotThreadSafe
class StatementCache {
    static final int DEFAULT_CACHE_SIZE = 0;
    static final int MINIMUM_CACHE_SIZE = 0;
    static final int MAXIMUM_CACHE_SIZE = 1000;
    static final int DEFAULT_TRACK_SIZE_MULTIPLIER = 10;
    static final int MAXIMUM_TRACK_SIZE = 2000;
    static final int DEFAULT_MAINTENANCE_FREQUENCY = 100;
    static final int MINIMUM_MAINTENANCE_FREQUENCY = 100;
    static final int MAXIMUM_MAINTENANCE_FREQUENCY = 1000;
    static final int DEFAULT_CACHE_COLD_MULTIPLIER = 2;
    static final int MAXIMUM_CACHE_COLD_MULTIPLIER = 10;
    static final int MINIMUM_CACHE_COLD_INTERVAL = 100;
    static final int DEFAULT_TRACK_COLD_MULTIPLIER = 5;
    static final int MAXIMUM_TRACK_COLD_MULTIPLIER = 25;
    static final int MINIMUM_TRACK_COLD_INTERVAL = 500;
    static final int MAX_TRACKED_STATEMENT_LENGTH = 10240;
    private final Tracer _tracer;
    private final ConnectionSapDB _connection;
    private final int _maxCachedSize;
    private final int _maxTrackedSize;
    private final int _maintenanceFrequency;
    private final int _cacheColdInterval;
    private final int _trackColdInterval;
    private final Map<String, ParseInfo> _cached;
    private final Map<String, TrackedInfo> _tracked;
    private final AtomicInteger _prepareCount;
    private final AtomicInteger _cacheHitCount;
    private final AtomicInteger _executeCount;
    private final AtomicInteger _dropCount;
    private final AtomicInteger _approxUniqueSQLTextCount;
    private final AtomicInteger _cacheRejectedFullCount;
    private final AtomicInteger _cacheEvictedFullCount;
    private final AtomicInteger _cacheEvictedColdCount;
    private final AtomicInteger _trackEvictedFullCount;
    private final AtomicInteger _trackEvictedColdCount;
    private final AtomicInteger _prepareCountAtLastColdCacheEviction;

    static StatementCache getInstance(ConnectionSapDB connection) {
        int trackColdInterval;
        int cacheColdInterval;
        int maintenanceFrequency;
        ConnectionProperties connectionProperties = connection.getConnectionProperties();
        int cacheSize = connectionProperties.getIntProperty(ConnectionProperty.STATEMENT_CACHE_SIZE);
        if ((cacheSize = StatementCache._constrainToRange(cacheSize, 0, 1000)) == 0) {
            return null;
        }
        int trackSize = connectionProperties.hasProperty(ConnectionProperty.STATEMENT_CACHE_TRACK_SIZE) ? connectionProperties.getIntProperty(ConnectionProperty.STATEMENT_CACHE_TRACK_SIZE) : cacheSize * 10;
        trackSize = StatementCache._constrainToRange(trackSize, cacheSize, 2000);
        if (connectionProperties.hasProperty(ConnectionProperty.STATEMENT_CACHE_MAINTENANCE_FREQUENCY)) {
            maintenanceFrequency = connectionProperties.getIntProperty(ConnectionProperty.STATEMENT_CACHE_MAINTENANCE_FREQUENCY);
            if (maintenanceFrequency != -1) {
                maintenanceFrequency = StatementCache._constrainToRange(maintenanceFrequency, 100, 1000);
            }
        } else {
            maintenanceFrequency = 100;
        }
        if (connectionProperties.hasProperty(ConnectionProperty.STATEMENT_CACHE_COLD_INTERVAL)) {
            cacheColdInterval = connectionProperties.getIntProperty(ConnectionProperty.STATEMENT_CACHE_COLD_INTERVAL);
            if (cacheColdInterval != -1) {
                cacheColdInterval = StatementCache._constrainToRange(cacheColdInterval, 100, cacheSize * 10);
            }
        } else {
            cacheColdInterval = Math.max(100, cacheSize * 2);
        }
        if (connectionProperties.hasProperty(ConnectionProperty.STATEMENT_CACHE_TRACK_COLD_INTERVAL)) {
            trackColdInterval = connectionProperties.getIntProperty(ConnectionProperty.STATEMENT_CACHE_TRACK_COLD_INTERVAL);
            if (trackColdInterval != -1) {
                trackColdInterval = StatementCache._constrainToRange(trackColdInterval, 500, trackSize * 25);
            }
        } else {
            trackColdInterval = Math.max(500, trackSize * 5);
        }
        return new StatementCache(connection.getTracer(), connection, cacheSize, trackSize, maintenanceFrequency, cacheColdInterval, trackColdInterval);
    }

    private static int _constrainToRange(int val, int min, int max) {
        if (val < min) {
            return min;
        }
        if (val > max) {
            return max;
        }
        return val;
    }

    private StatementCache(Tracer tracer, ConnectionSapDB connection, int maxCachedSize, int maxTrackedSize, int maintenanceFrequency, int cacheColdInterval, int trackColdInterval) {
        this._tracer = tracer;
        this._connection = connection;
        this._maxCachedSize = maxCachedSize;
        this._maxTrackedSize = maxTrackedSize;
        this._maintenanceFrequency = maintenanceFrequency;
        this._cacheColdInterval = cacheColdInterval;
        this._trackColdInterval = trackColdInterval;
        this._cached = new LinkedHashMap<String, ParseInfo>();
        this._tracked = new LinkedHashMap<String, TrackedInfo>();
        this._prepareCount = new AtomicInteger();
        this._cacheHitCount = new AtomicInteger();
        this._executeCount = new AtomicInteger();
        this._dropCount = new AtomicInteger();
        this._approxUniqueSQLTextCount = new AtomicInteger();
        this._cacheRejectedFullCount = new AtomicInteger();
        this._cacheEvictedFullCount = new AtomicInteger();
        this._cacheEvictedColdCount = new AtomicInteger();
        this._trackEvictedFullCount = new AtomicInteger();
        this._trackEvictedColdCount = new AtomicInteger();
        this._prepareCountAtLastColdCacheEviction = new AtomicInteger();
    }

    int getCurrentCacheSize() {
        return this._cached.size();
    }

    int getCurrentTrackSize() {
        return this._tracked.size();
    }

    int getPrepareCount() {
        return this._prepareCount.get();
    }

    int getCacheHitCount() {
        return this._cacheHitCount.get();
    }

    int getExecuteCount() {
        return this._executeCount.get();
    }

    int getDropCount() {
        return this._dropCount.get();
    }

    int getApproxUniqueSQLTextCount() {
        return this._approxUniqueSQLTextCount.get();
    }

    int getCacheRejectedFullCount() {
        return this._cacheRejectedFullCount.get();
    }

    int getCacheEvictedFullCount() {
        return this._cacheEvictedFullCount.get();
    }

    int getCacheEvictedColdCount() {
        return this._cacheEvictedColdCount.get();
    }

    int getTrackEvictedFullCount() {
        return this._trackEvictedFullCount.get();
    }

    int getTrackEvictedColdCount() {
        return this._trackEvictedColdCount.get();
    }

    ParseInfo onPrepareGetCached(String sql) {
        this._prepareCount.incrementAndGet();
        ParseInfo parseInfo = this._cached.remove(sql);
        if (parseInfo != null) {
            this._cacheHitCount.incrementAndGet();
            if (this._tracer.on()) {
                this._tracer.printStatementReused(parseInfo);
            }
        }
        return parseInfo;
    }

    void onPrepareSetTracked(ParseInfo parseInfo) throws SQLException {
        boolean wasFound;
        int connectionPrepareCount = this._prepareCount.get();
        int trackedSize = this._tracked.size();
        if (trackedSize == this._maxTrackedSize) {
            this._trackListColdHeuristic(connectionPrepareCount);
        }
        if (!this._isCandidate(parseInfo)) {
            return;
        }
        String sql = parseInfo.getSQL();
        TrackedInfo trackedInfo = this._tracked.remove(sql);
        boolean bl = wasFound = trackedInfo != null;
        if (!wasFound) {
            trackedInfo = new TrackedInfo();
            this._approxUniqueSQLTextCount.incrementAndGet();
        }
        trackedInfo._prepareCount++;
        trackedInfo._connectionPrepareCountAtLastPrepare = connectionPrepareCount;
        if (!wasFound && trackedSize == this._maxTrackedSize) {
            this._trackListFullHeuristic(trackedSize);
        }
        this._tracked.put(sql, trackedInfo);
    }

    void onExecute() throws SQLException {
        this._executeCount.incrementAndGet();
    }

    boolean onDrop(ParseInfo parseInfo) throws SQLException {
        this._dropCount.incrementAndGet();
        if (!this._isCandidate(parseInfo)) {
            return false;
        }
        String sql = parseInfo.getSQL();
        ParseInfo cachedParseInfo = this._cached.get(sql);
        if (cachedParseInfo != null) {
            if (parseInfo == cachedParseInfo) {
                throw new AssertionError((Object)"Cache corruption!");
            }
            return false;
        }
        TrackedInfo trackedInfo = this._tracked.get(sql);
        int connectionPrepareCount = this._prepareCount.get();
        if (trackedInfo == null) {
            return false;
        }
        trackedInfo._connectionPrepareCountAtLastDrop = connectionPrepareCount;
        if (this._cached.size() == this._maxCachedSize && !this._cacheFullHeuristic(trackedInfo._prepareCount) && !this._cacheColdHeuristic(connectionPrepareCount)) {
            return false;
        }
        this._cached.put(sql, parseInfo);
        if (this._tracer.on()) {
            this._tracer.printStatementCached(parseInfo);
        }
        return true;
    }

    void onSchemaChange() throws SQLException {
        this._cacheEvictAll();
    }

    void onInternalReconnect() throws SQLException {
        this._cached.clear();
    }

    void onCloseStatements(boolean sendRequests) throws SQLException {
        if (sendRequests) {
            this._cacheEvictAll();
        } else {
            this._cached.clear();
        }
        this._tracked.clear();
    }

    private boolean _isCandidate(ParseInfo parseInfo) {
        switch (parseInfo.getFunctionCode()) {
            case Insert: {
                if (parseInfo.getParameterCount() != 0) break;
                return false;
            }
        }
        return parseInfo.getSQL().length() <= 10240;
    }

    private boolean _cacheFullHeuristic(int prepareCount) throws SQLException {
        for (Map.Entry<String, ParseInfo> cachedEntry : this._cached.entrySet()) {
            TrackedInfo trackedInfo = this._tracked.get(cachedEntry.getKey());
            if (trackedInfo != null && (int)((double)trackedInfo._prepareCount * 1.5) >= prepareCount) continue;
            this._cacheEvict(cachedEntry);
            this._cacheEvictedFullCount.incrementAndGet();
            if (this._tracer.on()) {
                this._tracer.printStatementEvicted(cachedEntry.getValue(), "cache full");
            }
            return true;
        }
        this._cacheRejectedFullCount.incrementAndGet();
        return false;
    }

    private void _trackListFullHeuristic(int size) {
        String evictee = null;
        int idx = 0;
        int stopAt = (int)((double)size / 1.5);
        for (Map.Entry<String, TrackedInfo> trackedEntry : this._tracked.entrySet()) {
            if (idx++ == stopAt) break;
            String sql = trackedEntry.getKey();
            TrackedInfo trackedInfo = trackedEntry.getValue();
            if (this._cached.get(sql) != null || trackedInfo._prepareCount != 1) continue;
            evictee = sql;
            break;
        }
        if (evictee == null) {
            evictee = this._tracked.keySet().iterator().next();
        }
        this._tracked.remove(evictee);
        this._trackEvictedFullCount.incrementAndGet();
    }

    private boolean _cacheColdHeuristic(int connectionPrepareCount) throws SQLException {
        if (this._maintenanceFrequency == -1 || this._cacheColdInterval == -1 || connectionPrepareCount < this._prepareCountAtLastColdCacheEviction.get() + this._maintenanceFrequency) {
            return false;
        }
        Map.Entry<String, ParseInfo> cachedEntry = this._cached.entrySet().iterator().next();
        TrackedInfo trackedInfo = this._tracked.get(cachedEntry.getKey());
        if (trackedInfo == null || connectionPrepareCount - trackedInfo._connectionPrepareCountAtLastPrepare >= this._cacheColdInterval) {
            this._cacheEvict(cachedEntry);
            this._cacheEvictedColdCount.incrementAndGet();
            this._prepareCountAtLastColdCacheEviction.set(connectionPrepareCount);
            if (this._tracer.on()) {
                this._tracer.printStatementEvicted(cachedEntry.getValue(), "entry cold");
            }
            return true;
        }
        return false;
    }

    private void _trackListColdHeuristic(int connectionPrepareCount) {
        if (this._maintenanceFrequency == -1 || this._trackColdInterval == -1 || connectionPrepareCount % this._maintenanceFrequency != 0) {
            return;
        }
        for (Map.Entry<String, TrackedInfo> trackedEntry : this._tracked.entrySet()) {
            String sql = trackedEntry.getKey();
            TrackedInfo trackedInfo = trackedEntry.getValue();
            if (this._cached.get(sql) != null || trackedInfo._prepareCount <= 1 || connectionPrepareCount - trackedInfo._connectionPrepareCountAtLastDrop < this._trackColdInterval) continue;
            this._tracked.remove(sql);
            this._trackEvictedColdCount.incrementAndGet();
            break;
        }
    }

    private void _cacheEvict(Map.Entry<String, ParseInfo> entry) throws SQLException {
        this._connection._dropParseIDs(entry.getValue());
        this._cached.remove(entry.getKey());
    }

    private void _cacheEvictAll() throws SQLException {
        for (Map.Entry<String, ParseInfo> cachedEntry : new HashMap<String, ParseInfo>(this._cached).entrySet()) {
            this._cacheEvict(cachedEntry);
        }
    }

    boolean isCached(String sql) {
        return this._cached.containsKey(sql);
    }

    boolean isCached(ParseInfo parseInfo) {
        return this._cached.containsValue(parseInfo);
    }

    boolean isTracked(String sql) {
        return this._tracked.containsKey(sql);
    }

    private static class TrackedInfo {
        private int _prepareCount;
        private int _connectionPrepareCountAtLastPrepare;
        private int _connectionPrepareCountAtLastDrop;

        private TrackedInfo() {
        }

        public String toString() {
            return this._prepareCount + ":" + this._connectionPrepareCountAtLastPrepare + ":" + this._connectionPrepareCountAtLastDrop;
        }
    }
}

