package com.huifu.adapay.core.net;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.huifu.adapay.core.AdapayCore;
import com.huifu.adapay.core.exception.BaseAdaPayException;
import com.huifu.adapay.core.exception.FailureCode;
import com.huifu.adapay.core.util.AdapaySign;
import com.huifu.adapay.core.util.HttpClientUtils;
import com.huifu.adapay.core.util.ServerTypeEnum;

import java.io.File;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.config.RequestConfig;


/**
 * @author will.wang
 */
public abstract class AbstractRequest {

    private static final String AUTHORIZATION = "Authorization";
    private static final String SIGNATURE = "Signature";
    private static final String SDK_VERSION = "SDK_Version";
    private static final String API_NAME = "API_Name";
    private static final String SEQ_ID = "Seq_ID";
    private static final String APP_ID = "App_ID";
    private static final int CONNECTION_REQUEST_TIMEOUT = 10000;
    private static final int SOCKET_TIMEOUT = 30000;
    private static final int CONNECT_TIMEOUT = 30000;

    protected enum RequestMethod {
        /**
         * GET
         */
        GET,
        /**
         * POST
         */
        POST,
        /**
         * DELETE
         */
        DELETE,
        /**
         * PUT
         */
        PUT
    }

    /**
     * 与AdaPay服务器通讯并获取结果
     */
    protected static Map<String, Object> requestAdapayCore(RequestMethod method,
                                                           String uri,
                                                           Map<String, Object> params,
                                                           String privateKey,
                                                           String apiKey) throws BaseAdaPayException {

        return requestAdapayCore(method,
                uri,
                params,
                null,
                null,
                privateKey,
                apiKey,
                null);
    }


    protected static Map<String, Object> requestAdapayCore(RequestMethod method,
                                                           String uri,
                                                           Map<String, Object> params,
                                                           String privateKey,
                                                           String apiKey,
                                                           ServerTypeEnum serverType) throws BaseAdaPayException {

        return requestAdapayCore(method,
                uri,
                params,
                null,
                null,
                privateKey,
                apiKey,
                serverType);
    }

    /**
     * 与AdaPay服务器通讯并获取结果
     *
     * @param method 请求方式 RequestMethod
     * @param uri    请求url
     * @param params 请求参数

     * @return 返回对象
     * @throws BaseAdaPayException 异常
     */
    protected static Map<String, Object> requestAdapayCore(RequestMethod method,
                                                           String uri,
                                                           Map<String, Object> params,
                                                           File file,
                                                           String fileParam,
                                                           String privateKey,
                                                           String apiKey,
                                                           ServerTypeEnum serverType) throws BaseAdaPayException {
        String baseUrl = AdapayCore.apiBase;
        if (ServerTypeEnum.PAGE.equals(serverType)) {
            baseUrl = AdapayCore.pageBase;
        }


        StringBuilder signRequestUrl = new StringBuilder(baseUrl).append(uri);

        StringBuilder requestUrl = signRequestUrl;
        String region = (String) params.get("adapay_region");
        if (StringUtils.isNotBlank(region) &&
                region.equals("beijing") &&
                !ServerTypeEnum.PAGE.equals(serverType)&& "/v1/payments".equals(uri)){
            requestUrl = new StringBuilder("https://api-beijing.adapay.tech").append(uri);
        }

        RequestConfig userRequestConfig;
        int userConnectionRequestTimeout = (int) params.getOrDefault("adapay_connection_request_timeout", CONNECTION_REQUEST_TIMEOUT);
        int userConnectTimeout = (int) params.getOrDefault("adapay_connect_timeout", CONNECT_TIMEOUT);
        int userSocketTimeout = (int) params.getOrDefault("adapay_socket_timeout", SOCKET_TIMEOUT);
        userRequestConfig = RequestConfig.custom()
                .setSocketTimeout(userSocketTimeout)
                .setConnectTimeout(userConnectTimeout)
                .setConnectionRequestTimeout(userConnectionRequestTimeout).build();

        String back = null;
        Map<String, String> headers = new HashMap<>(2);

        String originalStr = "";

        if (RequestMethod.POST == method && file == null) {
            originalStr = params.isEmpty() ? "" : JSONObject.toJSONString(params);
        } else if (RequestMethod.POST == method && file != null) {
            originalStr = params.isEmpty() ? "" : AdapaySign.getOriginalStr(params);
        } else if (RequestMethod.GET == method) {
            originalStr = params.isEmpty() ? "" : AdapaySign.getOriginalStr(params);
        }

        if (AdapayCore.debug) {
            System.out.println("request string=" + originalStr);
        }
        String requestSign;
        try {
            requestSign = AdapaySign.sign(signRequestUrl + originalStr, privateKey);
        } catch (Exception e) {
            requestSign = null;
        }
        if (requestSign == null || requestSign.length() == 0) {
            throw new BaseAdaPayException(FailureCode.SECURITY_EXCEPTION.getFailureCode(),
                    "Sign error. Please check your privateKey.");
        }


        //设置请求头
        headers.put(AUTHORIZATION, apiKey);
        headers.put(SIGNATURE, requestSign);
        headers.put(SDK_VERSION, AdapayCore.sdkVersion);
        headers.put(API_NAME,uri);
        String seq_id = "seq" + UUID.randomUUID();
        headers.put(SEQ_ID, seq_id);


        if (AdapayCore.debug) {
            System.out.println("request apiKey=" + apiKey);
            System.out.println("request SIGNATURE=" + requestSign);
            System.out.println("request SDK_VERSION=" + AdapayCore.sdkVersion);
            System.out.println("request API_NAME" +  uri);
            System.out.println("request SEQ_ID" +  seq_id);
        }

        if (RequestMethod.POST == method && file == null) {
            headers.put("Content-type", "application/json");
            boolean isPaymentRequest = false;
            // 判断本次请求是不是除反扫外的支付下单请求
            // 支付接口使用独立的连接池
            String url = requestUrl.toString();
            if(url.contains("/v1")){
                String funcCode = url.split("/v1")[1];
                if(funcCode.equals("/payments")){
                    // 拿到 pay_channel
                    String pay_channel = (String) params.getOrDefault("pay_channel", "is scan");
                    // 判断是不是反扫交易
                    if (!pay_channel.contains("scan")){
                        isPaymentRequest = true;
                    }
                }
            }
            back = HttpClientUtils.httpPostJson(requestUrl.toString(),
                    headers,
                    originalStr,
                    userRequestConfig,
                    isPaymentRequest);
        } else if (RequestMethod.POST == method && file != null) {
            back = HttpClientUtils.httpPostFile(requestUrl.toString(),
                    headers,
                    params,
                    file,
                    fileParam,
                    userRequestConfig);
        } else if (RequestMethod.GET == method) {
            back = HttpClientUtils.httpGet(requestUrl.toString(),
                    headers,
                    params,
                    userRequestConfig,
                    false);
        }
        if (AdapayCore.debug) {
            System.out.println("post back=" + back);
        }
        if (back == null || back.length() == 0) {
            throw new BaseAdaPayException(FailureCode.SYSTEM_EXCEPTION.getFailureCode(),
                    "Request no response. Please check your network.");
        }
        JSONObject jsonObject = JSON.parseObject(back);
        String data = jsonObject.getString("data");
        String signature = jsonObject.getString("signature");

        boolean checkSign;
        try {
            String publicKey = AdapayCore.PUBLIC_KEY;

            checkSign = AdapaySign.verifySign(data, signature, publicKey);
        } catch (Exception e) {
            checkSign = false;
        }
        if (!checkSign) {
            if (AdapayCore.debug) {
                System.out.println("PUBLIC_KEY:" + AdapayCore.PUBLIC_KEY);
            }
            throw new BaseAdaPayException(FailureCode.SECURITY_EXCEPTION.getFailureCode(),
                    "Response sign check error. Please check your pubkey.");
        }
        try {
            JSONObject dataJson = JSONObject.parseObject(data);
            Map<String, Object> dataMap = (Map<String, Object>) dataJson;
            return dataMap;
        } catch (Exception e) {
            throw new BaseAdaPayException(FailureCode.SYSTEM_EXCEPTION.getFailureCode(),
                    "Parse response error." + e.getMessage());
        }
    }
}
