package com.huifu.adapay.core.util;

import org.apache.commons.codec.binary.Base64;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StringWriter;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * @author adapay.com
 */
public class AdapaySign {

    public AdapaySign() {
    }

    /**
     * 签名，固定UTF-8编码
     * @param content 待签名数据
     * @param privateKey 私钥数据
     * @return 签名结果
     * @throws Exception 异常
     */
    public static String sign(String content, String privateKey) throws Exception {
        return AdapaySign.sign(content, privateKey, "UTF-8");
    }

    /**
     * 验证签名，固定UTF-8编码
     * @param content 待验证数据
     * @param sign 待验证签名
     * @param publicKey 公钥数据
     * @return 验证结果
     * @throws Exception 异常
     */
    public static boolean verifySign(String content, String sign, String publicKey) throws Exception {
        return AdapaySign.verifySign(content, sign, publicKey, "UTF-8");
    }

    /**
     * 签名，指定编码
     * @param content 待签名数据
     * @param privateKey 私钥数据
     * @param charset 编码
     * @return 签名结果
     * @throws Exception 异常
     */
    public static String sign(String content, String privateKey, String charset) throws Exception {
        try {
            PrivateKey priKey = getPrivateKeyFromPKCS8("RSA", new ByteArrayInputStream(privateKey.getBytes()));
            Signature signature = Signature.getInstance("SHA1withRSA");
            signature.initSign(priKey);
            if (charset==null||charset.length()==0) {
                signature.update(content.getBytes());
            } else {
                signature.update(content.getBytes(charset));
            }

            byte[] signed = signature.sign();
            return new String(Base64.encodeBase64(signed));
        } catch (Exception var6) {
            throw new Exception("RSAcontent = " + content + "; charset = " + charset, var6);
        }
    }

    /**
     * 验证签名，指定编码
     * @param content 待验证数据
     * @param sign 待验证签名
     * @param publicKey 公钥数据
     * @param charset 编码
     * @return 验证结果
     * @throws Exception 异常
     */
    public static boolean verifySign(String content, String sign, String publicKey, String charset) throws Exception {
        try {
            PublicKey pubKey = getPublicKeyFromX509("RSA", new ByteArrayInputStream(publicKey.getBytes()));
            Signature signature = Signature.getInstance("SHA1withRSA");
            signature.initVerify(pubKey);
            if (charset==null||charset.length()==0) {
                signature.update(content.getBytes());
            } else {
                signature.update(content.getBytes(charset));
            }

            return signature.verify(Base64.decodeBase64(sign.getBytes()));
        } catch (Exception var6) {
            throw new Exception("RSAcontent = " + content + ",sign=" + sign + ",charset = " + charset, var6);
        }
    }

    /**
     * 生成私钥
     * @param algorithm 签名算法
     * @param ins 私钥数据
     * @return 私钥
     * @throws Exception 异常
     */
    public static PrivateKey getPrivateKeyFromPKCS8(String algorithm, InputStream ins) throws Exception {
        boolean validKey = (ins != null && algorithm!=null ||algorithm.length()!=0);
        if (validKey) {
            KeyFactory keyFactory = KeyFactory.getInstance(algorithm);
            byte[] encodedKey = StreamUtil.readText(ins).getBytes();
            encodedKey = Base64.decodeBase64(encodedKey);
            return keyFactory.generatePrivate(new PKCS8EncodedKeySpec(encodedKey));
        } else {
            return null;
        }
    }

    /**
     * 生成公钥
     * @param algorithm 签名算法
     * @param ins 公钥数据
     * @return 公钥
     * @throws Exception 异常
     */
    public static PublicKey getPublicKeyFromX509(String algorithm, InputStream ins) throws Exception {
        KeyFactory keyFactory = KeyFactory.getInstance(algorithm);
        StringWriter writer = new StringWriter();
        StreamUtil.io(new InputStreamReader(ins), writer);
        byte[] encodedKey = writer.toString().getBytes();
        encodedKey = Base64.decodeBase64(encodedKey);
        return keyFactory.generatePublic(new X509EncodedKeySpec(encodedKey));
    }

    /**
     * 根据签名规则将请求参数转换为明文
     * @param map 请求参数
     * @return 数据明文
     */
    public static String getOriginalStr(Map<String, Object> map) {
        List<String> listKeys = new ArrayList<>(map.keySet());
        Collections.sort(listKeys);
        StringBuilder stringBuilder = new StringBuilder();
        for (int i = 0; i < listKeys.size(); i++) {
            if (map.get(listKeys.get(i)) == null || map.get(listKeys.get(i)).toString().length() == 0) {
                continue;
            }
            stringBuilder.append(listKeys.get(i)).append("=").append(map.get(listKeys.get(i))).append("&");
        }
        return stringBuilder.length() == 0 ? "" : stringBuilder.toString().substring(0, stringBuilder.length() - 1);
    }
}
