package com.huifu.adapay.core.util;

import com.huifu.adapay.core.exception.BaseAdaPayException;
import com.huifu.adapay.core.exception.FailureCode;
import org.apache.http.*;
import org.apache.http.client.HttpRequestRetryHandler;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.ConnectTimeoutException;
import org.apache.http.conn.routing.HttpRoute;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.LayeredConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLContexts;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.FileBody;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.protocol.HttpContext;
import org.apache.http.util.EntityUtils;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLHandshakeException;
import javax.net.ssl.X509TrustManager;
import java.io.*;
import java.net.SocketTimeoutException;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.security.KeyManagementException;
import java.security.cert.X509Certificate;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;

/**
 * 基于HttpClient实现的Http请求工具
 * 支持POST和GET请求, 支持SSL
 *
 * @author huifu.com
 */
public class HttpClientUtils {
    private static final int CONNECTION_REQUEST_TIMEOUT = 10000;
    private static final int SOCKET_TIMEOUT = 30000;
    private static final int CONNECT_TIMEOUT = 30000;

    private static final ReentrantLock lock = new ReentrantLock();
    /**
     * 编码
     */
    private static final String ENCODING = "UTF-8";
    private static int connectionRequestTimeout = 30000;
    private static int socketTimeout = 20000;
    private static int connectTimout = 20000;
    private static CloseableHttpClient httpClient;
    private static CloseableHttpClient paymentHttpClient;

    private static RequestConfig requestConfig;

    static {
        init();
    }

    static void init() {
        initRequestConfig();
    }

    static void initRequestConfig() {
        requestConfig = RequestConfig.custom()
                .setSocketTimeout(socketTimeout)
                .setConnectTimeout(connectTimout)
                .setConnectionRequestTimeout(connectionRequestTimeout).build();
    }

    /**
     * 获取HttpClient 使用默认config
     *
     * @param url
     * @return
     */
    public static CloseableHttpClient getHttpClient(String url) {
        return getHttpClient(url, requestConfig);
    }

    /**
     * 获取HttpClient带config
     *
     * @param url
     * @param config
     * @return
     */
    public static CloseableHttpClient getHttpClient(String url, RequestConfig config) {
        return getHttpClient(url, config, false);
    }
    /**
     * 获取HttpClient带config
     *
     * @param url
     * @param config
     * @return
     */
    public static CloseableHttpClient getHttpClient(String url, RequestConfig config, boolean isPaymentRequest) {
        String hostname = url.split("/")[2];
        int port = 80;
        if (hostname.contains(":")) {
            String[] arr = hostname.split(":");
            hostname = arr[0];
            if (arr.length == 2) {
                port = Integer.parseInt(arr[1]);
            }
        }
        if (isPaymentRequest) {
            if (paymentHttpClient == null) {
                lock.lock();
                try {
                    if (paymentHttpClient == null) {
                        paymentHttpClient = createHttpClient(500, 40, 100, hostname, port, config);
                    }
                } finally {
                    lock.unlock();
                }
            }
            return paymentHttpClient;
        } else {
            if (httpClient == null) {
                lock.lock();
                try {
                    if (httpClient == null) {
                        httpClient = createHttpClient(500, 40, 100, hostname, port, config);
                    }
                } finally {
                    lock.unlock();
                }
            }
            return httpClient;
        }
    }

    /**
     * 创建 http 请求客户端 附带默认requestConfig
     *
     * @param maxTotal
     * @param maxPerRoute
     * @param maxRoute
     * @param hostname
     * @param port
     * @return
     */
    public static CloseableHttpClient createHttpClient(int maxTotal,
                                                       int maxPerRoute,
                                                       int maxRoute,
                                                       String hostname,
                                                       int port) {
        return createHttpClient(maxTotal, maxPerRoute, maxRoute, hostname, port, requestConfig);
    }

    /**
     * 创建连接池
     * @param maxTotal
     * @param maxPerRoute
     * @param maxRoute
     * @param hostname
     * @param port
     * @return
     */
    public static PoolingHttpClientConnectionManager createPoolingHttpClientConnectionManager(int maxTotal,
                                                                                              int maxPerRoute,
                                                                                              int maxRoute,
                                                                                              String hostname,
                                                                                              int port){
        ConnectionSocketFactory plainsf = PlainConnectionSocketFactory.getSocketFactory();
        SSLContext sslcontext = SSLContexts.createDefault();
        javax.net.ssl.TrustManager[] trustAllCerts = new javax.net.ssl.TrustManager[1];
        X509TrustManager tm = getX509TrustManager();
        trustAllCerts[0] = tm;
        try {
            sslcontext.init(null, trustAllCerts, null);
        } catch (KeyManagementException e) {
        }
        LayeredConnectionSocketFactory sslsf = new SSLConnectionSocketFactory(sslcontext,
                SSLConnectionSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER);

        Registry<ConnectionSocketFactory> registry = RegistryBuilder
                .<ConnectionSocketFactory>create().register("http", plainsf)
                .register("https", sslsf).build();
        PoolingHttpClientConnectionManager cm = new PoolingHttpClientConnectionManager(registry, null, null, null, 5,
                TimeUnit.MINUTES);
        // 连接池最大并发数
        cm.setMaxTotal(maxTotal);
        // 单路由最大并发量，即单个接口的最大并发量(单台服务器)
        cm.setDefaultMaxPerRoute(maxPerRoute);
        HttpHost httpHost = new HttpHost(hostname, port);
        // 将目标主机的最大连接数增加
        cm.setMaxPerRoute(new HttpRoute(httpHost), maxRoute);
        return cm;
    }

    /**
     * 创建请求重试 Handler
     * @return
     */
    public static HttpRequestRetryHandler createHttpRequestRetryHandler(){
        // 请求重试处理
        return (exception, executionCount, context) -> {
            if (executionCount >= 3) {// 如果已经重试了3次，就放弃
                return false;
            }
            if (exception instanceof NoHttpResponseException) {// 如果服务器丢掉了连接，那么就重试
                return true;
            }
            if (exception instanceof SSLHandshakeException) {// 不要重试SSL握手异常
                return false;
            }
            if (exception instanceof InterruptedIOException) {// 超时
                return false;
            }
            if (exception instanceof UnknownHostException) {// 目标服务器不可达
                return false;
            }
            if (exception instanceof SSLException) {// SSL握手异常
                return false;
            }

            HttpClientContext clientContext = HttpClientContext
                    .adapt(context);
            HttpRequest request = clientContext.getRequest();
            // 如果请求是幂等的，就再次尝试
            if (!(request instanceof HttpEntityEnclosingRequest)) {
                return true;
            }
            return false;
        };
    }

    /**
     * 创建 http 请求客户端
     *
     * @param maxTotal
     * @param maxPerRoute
     * @param maxRoute
     * @param hostname
     * @param port
     * @param config
     * @return
     */
    public static CloseableHttpClient createHttpClient(int maxTotal,
                                                       int maxPerRoute,
                                                       int maxRoute,
                                                       String hostname,
                                                       int port,
                                                       RequestConfig config) {

        PoolingHttpClientConnectionManager cm = createPoolingHttpClientConnectionManager(maxTotal,
                maxPerRoute, maxRoute, hostname, port);

        // 请求重试处理
        HttpRequestRetryHandler httpRequestRetryHandler = createHttpRequestRetryHandler();

        CloseableHttpClient httpClient = HttpClients.custom()
                .setConnectionManager(cm).setDefaultRequestConfig(config)
                .setRetryHandler(httpRequestRetryHandler)
                .build();

        closeExpiredConnectionsPeriodTask(1, cm);

        return httpClient;
    }

    public static X509TrustManager getX509TrustManager() {
        return new X509TrustManager() {
            @Override
            public void checkClientTrusted(X509Certificate[] arg0, String arg1) {
            }

            @Override
            public void checkServerTrusted(X509Certificate[] arg0, String arg1) {
            }

            @Override
            public X509Certificate[] getAcceptedIssuers() {
                return null;
            }
        };
    }

    private static void closeExpiredConnectionsPeriodTask(final int timeUnitBySecond, final PoolingHttpClientConnectionManager cm) {
        ScheduledExecutorService executorService = Executors.newScheduledThreadPool(1);
        executorService.scheduleAtFixedRate(new Runnable() {
            @Override
            public void run() {
                if (cm != null) {
                    cm.closeExpiredConnections();
                }
            }
        }, 10, timeUnitBySecond, TimeUnit.MINUTES);
    }


    /**
     * post请求,支持SSL
     *
     * @param url    请求地址
     * @param params 请求参数
     * @return 响应信息
     * @throws BaseAdaPayException 异常
     */
    public static String httpPost(String url, Map<String, Object> params, boolean isJson) throws BaseAdaPayException {
        RequestConfig userRequestConfig =  RequestConfig.custom()
                .setConnectionRequestTimeout(CONNECTION_REQUEST_TIMEOUT)
                .setConnectTimeout(CONNECT_TIMEOUT)
                .setSocketTimeout(SOCKET_TIMEOUT).build();
        return httpPost(url, null, params, false, null, isJson, null, null, userRequestConfig);
    }

    /**
     * post请求,支持SSL
     *
     * @param url    请求地址
     * @param params 请求参数
     * @return 响应信息
     * @throws BaseAdaPayException 异常
     */
    public static String httpPost(String url, Map<String, String> headers, Map<String, Object> params) throws BaseAdaPayException {
        RequestConfig userRequestConfig =  RequestConfig.custom()
                .setConnectionRequestTimeout(CONNECTION_REQUEST_TIMEOUT)
                .setConnectTimeout(CONNECT_TIMEOUT)
                .setSocketTimeout(SOCKET_TIMEOUT).build();
        return httpPost(url, headers, params, false, null, false, null, null, userRequestConfig);
    }

    /**
     * httpPostJson whit isPaymentRequest
     * @param url 请求url
     * @param headers headers
     * @param json json参数
     * @param config RequestConfig
     * @param isPaymentRequest 是否是支付请求，scan除外
     * @return 结果string
     * @throws BaseAdaPayException
     */
    public static String httpPostJson(String url,
                                      Map<String, String> headers,
                                      String json,
                                      RequestConfig config,
                                      boolean isPaymentRequest) throws BaseAdaPayException {
        return httpPost(url,
                headers, null, false, json, true, null, null, config, isPaymentRequest);
    }

    /**
     * post请求,支持SSL
     *
     * @param url  请求地址
     * @param json 请求参数
     * @return 响应信息
     * @throws BaseAdaPayException 异常
     */
    public static String httpPostJson(String url, Map<String, String> headers, String json, RequestConfig config) throws BaseAdaPayException {
        return httpPost(url, headers, null, false, json, true, null, null, config);
    }

    /**
     * post请求,支持SSL
     *
     * @param url    请求地址
     * @param params 请求参数
     * @return 响应信息
     * @throws BaseAdaPayException 异常
     */
    public static String httpPostFile(String url, Map<String, String> headers, Map<String, Object> params, File file, String fileParam, RequestConfig config) throws BaseAdaPayException {
        return httpPost(url, headers, params, false, null, false, file, fileParam, config);
    }

    /**
     * post请求,支持SSL
     *
     * @param url                      请求地址
     * @param params                   请求参数
     * @param connectionRequestTimeout 超时时间(毫秒):从连接池获取连接的时间
     * @param connectTimeout           超时时间(毫秒):请求时间
     * @param socketTimeout            超时时间(毫秒):响应时间
     * @return 响应信息
     * @throws BaseAdaPayException 异常
     */
    public static String httpPost(String url, Map<String, Object> params, int connectionRequestTimeout, int connectTimeout, int socketTimeout, String json, boolean isJson) throws BaseAdaPayException {
        RequestConfig userRequestConfig =  RequestConfig.custom()
                .setConnectionRequestTimeout(CONNECTION_REQUEST_TIMEOUT)
                .setConnectTimeout(CONNECT_TIMEOUT)
                .setSocketTimeout(SOCKET_TIMEOUT).build();
        return httpPost(url, null, params, false, json, isJson, null, null, userRequestConfig);
    }

    /**
     * post请求,支持SSL
     *
     * @param url      请求地址
     * @param headers  请求头信息
     * @param params   请求参数
     * @param isStream 是否以流的方式获取响应信息
     * @param isJson   json
     * @return 响应信息
     * @throws BaseAdaPayException 异常
     */
    public static String httpPost(String url,
                                  Map<String, String> headers,
                                  Map<String, Object> params,
                                  boolean isStream,
                                  String json,
                                  boolean isJson,
                                  File file,
                                  String fileParam,
                                  RequestConfig userRequestConfig) throws BaseAdaPayException {
        return httpPost(url, headers, params, isStream, json, isJson, file, fileParam, userRequestConfig, false);

    }
    /**
     * post请求,支持SSL
     *
     * @param url      请求地址
     * @param headers  请求头信息
     * @param params   请求参数
     * @param isStream 是否以流的方式获取响应信息
     * @param isJson   json
     * @return 响应信息
     * @throws BaseAdaPayException 异常
     */
    public static String httpPost(String url,
                                  Map<String, String> headers,
                                  Map<String, Object> params,
                                  boolean isStream,
                                  String json,
                                  boolean isJson,
                                  File file,
                                  String fileParam,
                                  RequestConfig userRequestConfig, boolean isPaymentRequest) throws BaseAdaPayException {

        // 创建post请求
        HttpPost httpPost = new HttpPost(url);


        // 添加请求头信息
        if (null != headers) {
            for (Map.Entry<String, String> entry : headers.entrySet()) {
                httpPost.addHeader(entry.getKey(), entry.getValue());
            }
        }
        if (file != null) {
            MultipartEntityBuilder builder = MultipartEntityBuilder.create();
            builder.setCharset(java.nio.charset.Charset.forName("UTF-8"));
            builder.setMode(HttpMultipartMode.BROWSER_COMPATIBLE);
            // 文件流

            builder.addPart(fileParam, new FileBody(file));
            //决中文乱码
            for (Map.Entry<String, Object> entry : params.entrySet()) {
                if (entry.getValue() == null) {
                    continue;
                }
                // 类似浏览器表单提交，对应input的name和value
                builder.addTextBody(entry.getKey(), entry.getValue().toString(), ContentType.APPLICATION_FORM_URLENCODED.withCharset("utf-8"));
            }
            httpPost.setEntity(builder.build());
        }
        // 添加请求参数信息
        if (file == null && null != params) {
            try {
                httpPost.setEntity(new UrlEncodedFormEntity(covertParams2NVPS(params), ENCODING));
            } catch (UnsupportedEncodingException e) {
                e.printStackTrace();
                throw new BaseAdaPayException(FailureCode.SYSTEM_EXCEPTION);
            }
        }


        if (isJson && json != null && json.length() != 0) {
            StringEntity entity = new StringEntity(json, ENCODING);
            entity.setContentEncoding(ENCODING);
            entity.setContentType("application/json");
            httpPost.setEntity(entity);
        }
        // 判断本次请求是不是除反扫外的支付下单请求
        // 支付接口使用独立的连接池
        if(isPaymentRequest){
            return getResult(httpPost, url, isStream, userRequestConfig, true);
        }

        return getResult(httpPost, url, isStream, userRequestConfig);
    }

    /**
     * get请求,支持SSL
     *
     * @param url    请求地址
     * @param params 请求参数
     * @return 响应信息
     * @throws BaseAdaPayException 异常
     */
    public static String httpGet(String url,
                                 Map<String, Object> params) throws BaseAdaPayException {

        RequestConfig userRequestConfig;
        if (params != null) {
            int userConnectionRequestTimeout = (int) params.getOrDefault("adapay_connection_request_timeout",
                    CONNECTION_REQUEST_TIMEOUT);
            int userConnectTimeout = (int) params.getOrDefault("adapay_connect_timeout", CONNECT_TIMEOUT);
            int userSocketTimeout = (int) params.getOrDefault("adapay_socket_timeout", SOCKET_TIMEOUT);
            userRequestConfig = RequestConfig.custom()
                    .setConnectionRequestTimeout(userConnectionRequestTimeout)
                    .setConnectTimeout(userConnectTimeout)
                    .setSocketTimeout(userSocketTimeout).build();

        } else {
            userRequestConfig = RequestConfig.custom()
                    .setConnectionRequestTimeout(CONNECTION_REQUEST_TIMEOUT)
                    .setConnectTimeout(CONNECT_TIMEOUT)
                    .setSocketTimeout(SOCKET_TIMEOUT).build();

        }
        return httpGet(url,
                null,
                params,
                userRequestConfig,
                false);
    }

    /**
     * get请求,支持SSL
     *
     * @param url    请求地址
     * @param params 请求参数
     * @return 响应信息
     * @throws BaseAdaPayException 异常
     */
    public static String httpGet(String url,
                                 Map<String, String> headers,
                                 Map<String, Object> params) throws BaseAdaPayException {
        RequestConfig userRequestConfig;
        if (params != null) {
            int userConnectionRequestTimeout = (int) params.getOrDefault("adapay_connection_request_timeout",
                    CONNECTION_REQUEST_TIMEOUT);
            int userConnectTimeout = (int) params.getOrDefault("adapay_connect_timeout", CONNECT_TIMEOUT);
            int userSocketTimeout = (int) params.getOrDefault("adapay_socket_timeout", SOCKET_TIMEOUT);
            userRequestConfig = RequestConfig.custom()
                    .setConnectionRequestTimeout(userConnectionRequestTimeout)
                    .setConnectTimeout(userConnectTimeout)
                    .setSocketTimeout(userSocketTimeout).build();

        } else {
            userRequestConfig = RequestConfig.custom()
                    .setConnectionRequestTimeout(CONNECTION_REQUEST_TIMEOUT)
                    .setConnectTimeout(CONNECT_TIMEOUT)
                    .setSocketTimeout(SOCKET_TIMEOUT).build();

        }
        return httpGet(url,
                headers,
                params,
                userRequestConfig,
                false);
    }

    /**
     * get请求,支持SSL
     *
     * @param url                      请求地址
     * @param params                   请求参数
     * @return 响应信息
     * @throws BaseAdaPayException 异常
     */
    public static String httpGet(String url,
                                 Map<String, Object> params,
                                 RequestConfig config) throws BaseAdaPayException {
        return httpGet(url,
                null,
                params,
                config,
                false);
    }

    /**
     * get请求,支持SSL
     *
     * @param url                           请求地址
     * @param headers                       请求头信息
     * @param params                        请求参数
     * @param isStream                      是否以流的方式获取响应信息
     * @return 响应信息
     * @throws BaseAdaPayException 异常
     */
    public static String httpGet(String url,
                                 Map<String, String> headers,
                                 Map<String, Object> params, RequestConfig config,
                                 boolean isStream) throws BaseAdaPayException {
        // 构建url
        URIBuilder uriBuilder;
        try {
            uriBuilder = new URIBuilder(url);
        } catch (URISyntaxException e) {
            e.printStackTrace();
            throw new BaseAdaPayException(FailureCode.SYSTEM_EXCEPTION);
        }
        // 添加请求参数信息
        if (null != params) {
            uriBuilder.setParameters(covertParams2NVPS(params));
        }

        // 创建post请求
        HttpGet httpGet;
        try {
            httpGet = new HttpGet(uriBuilder.build());
        } catch (Exception e) {
            throw new BaseAdaPayException(FailureCode.SYSTEM_EXCEPTION);
        }

        // 添加请求头信息
        if (null != headers) {
            for (Map.Entry<String, String> entry : headers.entrySet()) {
                httpGet.addHeader(entry.getKey(), entry.getValue());
            }
        }
        return getResult(httpGet, url, isStream, config);
    }

    /**
     * 发送请求 默认requestConfig
     *
     * @param httpRequest 请求
     * @param url         url
     * @param isStream    是否以流形式获取
     * @return 响应结果
     */
    private static String getResult(HttpRequestBase httpRequest,
                                    String url,
                                    boolean isStream) throws BaseAdaPayException {
        return getResult(httpRequest, url, isStream, requestConfig);
    }

    /**
     * 发送请求 默认requestConfig
     *
     * @param httpRequest 请求
     * @param url         url
     * @param isStream    是否以流形式获取
     * @param config      requestConfig
     * @return
     * @throws BaseAdaPayException
     */
    private static String getResult(HttpRequestBase httpRequest,
                                    String url,
                                    boolean isStream,
                                    RequestConfig config) throws BaseAdaPayException {
        return getResult(httpRequest, url, isStream, config, false);

    }
    /**
     * 发送请求 默认requestConfig
     *
     * @param httpRequest 请求
     * @param url         url
     * @param isStream    是否以流形式获取
     * @param config      requestConfig
     * @return
     * @throws BaseAdaPayException
     */
    private static String getResult(HttpRequestBase httpRequest,
                                    String url,
                                    boolean isStream,
                                    RequestConfig config, boolean isPaymentRequest) throws BaseAdaPayException {
        // 响应结果
        StringBuilder sb = new StringBuilder();
        CloseableHttpResponse response = null;
        BufferedReader br = null;
        InputStreamReader is = null;
        try {
            // 获取客户端连接对象
            CloseableHttpClient httpClient = getHttpClient(url, config, isPaymentRequest);
            // 发起请求
            response = httpClient.execute(httpRequest);
            int respCode = response.getStatusLine().getStatusCode();

            // 正确响应
            if (HttpStatus.SC_OK == respCode) {
                // 获得响应实体
                HttpEntity entity = response.getEntity();
                // 如果是以流的形式获取
                if (isStream) {
                    is = new InputStreamReader(entity.getContent(), ENCODING);
                    br = new BufferedReader(is);
                    String len;
                    while ((len = br.readLine()) != null) {
                        sb.append(len);
                    }
                } else {
                    sb.append(EntityUtils.toString(entity, ENCODING));
                }
            } else {
                sb.append(response.getEntity() == null ? "" : EntityUtils.toString(response.getEntity(), ENCODING));
            }
        } catch (SocketTimeoutException | ConnectTimeoutException e){
            throw new BaseAdaPayException(FailureCode.HTTP_TIMEOUT_EXCEPTION);
        } catch (UnsupportedOperationException | ParseException | IOException e) {
            e.printStackTrace();
            throw new BaseAdaPayException(FailureCode.SYSTEM_EXCEPTION);
        } finally {
            try {
                if (null != br) {
                    br.close();
                }
                if (null != is) {
                    is.close();
                }
                if (null != response) {
                    response.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
                throw new BaseAdaPayException(FailureCode.SYSTEM_EXCEPTION);
            }
        }
        return sb.toString();
    }

    /**
     * Map转换成NameValuePair List集合
     *
     * @param params map
     * @return NameValuePair List集合
     */
    private static List<NameValuePair> covertParams2NVPS(Map<String, Object> params) {
        List<NameValuePair> paramList = new LinkedList<>();
        for (Map.Entry<String, Object> entry : params.entrySet()) {
            paramList.add(new BasicNameValuePair(entry.getKey(), null == entry.getValue() ? "" : entry.getValue().toString()));
        }
        return paramList;
    }

}