/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.ext.jwt;

import io.vertx.core.json.JsonObject;
import io.vertx.core.logging.Logger;
import io.vertx.core.logging.LoggerFactory;
import io.vertx.ext.jwt.Crypto;
import io.vertx.ext.jwt.CryptoKeyPair;
import io.vertx.ext.jwt.CryptoMac;
import io.vertx.ext.jwt.CryptoNone;
import io.vertx.ext.jwt.CryptoSignature;
import java.io.ByteArrayInputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.KeyFactory;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

public final class JWT {
    private static final List<String> PUBSEC_ALGS = Arrays.asList("RS256", "RS384", "RS512", "ES256", "ES384", "ES512");
    private static final List<String> MAC_ALGS = Arrays.asList("HS256", "HS384", "HS512");
    private static final Random RND = new Random();
    private static final Map<String, String> ALGORITHM_ALIAS = new HashMap<String, String>(){
        {
            this.put("HS256", "HMacSHA256");
            this.put("HS384", "HMacSHA384");
            this.put("HS512", "HMacSHA512");
            this.put("RS256", "SHA256withRSA");
            this.put("RS384", "SHA384withRSA");
            this.put("RS512", "SHA512withRSA");
            this.put("ES256", "SHA256withECDSA");
            this.put("ES384", "SHA384withECDSA");
            this.put("ES512", "SHA512withECDSA");
        }
    };
    private static final Charset UTF8 = StandardCharsets.UTF_8;
    private static final Logger log = LoggerFactory.getLogger(JWT.class);
    private static final JsonObject EMPTY = new JsonObject();
    private static final Base64.Encoder encoder = Base64.getUrlEncoder().withoutPadding();
    private static final Base64.Decoder decoder = Base64.getUrlDecoder();
    private final Map<String, List<Crypto>> cryptoMap = new HashMap<String, List<Crypto>>();
    private boolean unsecure = true;

    public JWT() {
        this.cryptoMap.put("none", Collections.singletonList(new CryptoNone()));
    }

    public JWT(KeyStore keyStore, char[] keyStorePassword) {
        this();
        for (String alg : Arrays.asList("HS256", "HS384", "HS512")) {
            try {
                Mac mac = this.getMac(keyStore, keyStorePassword, alg);
                if (mac != null) {
                    List l = this.cryptoMap.computeIfAbsent(alg, k -> new ArrayList());
                    l.add(new CryptoMac(mac));
                    continue;
                }
                log.info((Object)(alg + " not available"));
            }
            catch (RuntimeException e) {
                log.warn((Object)(alg + " not supported"), (Throwable)e);
            }
        }
        for (String alg : Arrays.asList("RS256", "RS384", "RS512", "ES256", "ES384", "ES512")) {
            try {
                X509Certificate certificate = this.getCertificate(keyStore, alg);
                PrivateKey privateKey = this.getPrivateKey(keyStore, keyStorePassword, alg);
                if (certificate != null && privateKey != null) {
                    List l = this.cryptoMap.computeIfAbsent(alg, k -> new ArrayList());
                    l.add(new CryptoSignature(ALGORITHM_ALIAS.get(alg), certificate, privateKey));
                    continue;
                }
                log.info((Object)(alg + " not available"));
            }
            catch (RuntimeException e) {
                e.printStackTrace();
                log.warn((Object)(alg + " not supported"));
            }
        }
        this.unsecure = this.cryptoMap.size() == 1;
    }

    @Deprecated
    public JWT(String key, boolean keyPrivate) {
        this();
        if (keyPrivate) {
            this.addSecretKey("RS256", key);
        } else {
            this.addPublicKey("RS256", key);
        }
    }

    private static KeyFactory getKeyFactoryFor(String algorithm) throws NoSuchAlgorithmException {
        switch (algorithm.charAt(0)) {
            case 'R': {
                return KeyFactory.getInstance("RSA");
            }
            case 'E': {
                return KeyFactory.getInstance("EC");
            }
        }
        throw new RuntimeException("Unknown algorithm factory for: " + algorithm);
    }

    public JWT addPublicKey(String algorithm, String key) {
        return this.addKeyPair(algorithm, key, null);
    }

    public JWT addKeyPair(String algorithm, String publicKey, String privateKey) {
        if (!PUBSEC_ALGS.contains(algorithm)) {
            throw new RuntimeException("Unknown algorithm: " + algorithm);
        }
        if (publicKey == null || privateKey == null) {
            this.cryptoMap.remove(algorithm);
            if (publicKey == null && privateKey == null) {
                this.unsecure = this.cryptoMap.size() == 1;
                return this;
            }
        }
        try {
            PrivateKey sec;
            PublicKey pub;
            if (publicKey != null) {
                KeyFactory pubkf = JWT.getKeyFactoryFor(algorithm);
                X509EncodedKeySpec pubspec = new X509EncodedKeySpec(Base64.getMimeDecoder().decode(publicKey));
                pub = pubkf.generatePublic(pubspec);
            } else {
                pub = null;
            }
            if (privateKey != null) {
                KeyFactory seckf = JWT.getKeyFactoryFor(algorithm);
                PKCS8EncodedKeySpec secspec = new PKCS8EncodedKeySpec(Base64.getMimeDecoder().decode(privateKey));
                sec = seckf.generatePrivate(secspec);
            } else {
                sec = null;
            }
            List l = this.cryptoMap.computeIfAbsent(algorithm, k -> new ArrayList());
            l.add(new CryptoKeyPair(ALGORITHM_ALIAS.get(algorithm), pub, sec));
        }
        catch (RuntimeException | NoSuchAlgorithmException | InvalidKeySpecException e) {
            throw new RuntimeException(algorithm + " not supported", e);
        }
        this.unsecure = this.cryptoMap.size() == 1;
        return this;
    }

    public JWT addSecretKey(String algorithm, String key) {
        return this.addKeyPair(algorithm, null, key);
    }

    public JWT addCertificate(String algorithm, String cert) {
        try {
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            X509Certificate certificate = (X509Certificate)cf.generateCertificate(new ByteArrayInputStream(cert.getBytes(UTF8)));
            List l = this.cryptoMap.computeIfAbsent(algorithm, k -> new ArrayList());
            l.add(new CryptoSignature(ALGORITHM_ALIAS.get(algorithm), certificate, null));
            this.unsecure = this.cryptoMap.size() == 1;
        }
        catch (CertificateException e) {
            throw new RuntimeException(e);
        }
        return this;
    }

    public JWT addSecret(String algorithm, String key) {
        if (!MAC_ALGS.contains(algorithm)) {
            throw new RuntimeException("Unknown algorithm: " + algorithm);
        }
        if (key == null) {
            this.cryptoMap.remove(algorithm);
            this.unsecure = this.cryptoMap.size() == 1;
            return this;
        }
        try {
            Mac mac = Mac.getInstance(ALGORITHM_ALIAS.get(algorithm));
            mac.init(new SecretKeySpec(key.getBytes(), ALGORITHM_ALIAS.get(algorithm)));
            List l = this.cryptoMap.computeIfAbsent(algorithm, k -> new ArrayList());
            l.add(new CryptoMac(mac));
        }
        catch (RuntimeException | InvalidKeyException | NoSuchAlgorithmException e) {
            throw new RuntimeException(algorithm + " not supported", e);
        }
        this.unsecure = this.cryptoMap.size() == 1;
        return this;
    }

    private Mac getMac(KeyStore keyStore, char[] keyStorePassword, String alias) {
        try {
            Key secretKey = keyStore.getKey(alias, keyStorePassword);
            if (secretKey == null) {
                return null;
            }
            Mac mac = Mac.getInstance(secretKey.getAlgorithm());
            mac.init(secretKey);
            return mac;
        }
        catch (InvalidKeyException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException e) {
            throw new RuntimeException(e);
        }
    }

    private X509Certificate getCertificate(KeyStore keyStore, String alias) {
        try {
            return (X509Certificate)keyStore.getCertificate(alias);
        }
        catch (KeyStoreException e) {
            throw new RuntimeException(e);
        }
    }

    private PrivateKey getPrivateKey(KeyStore keyStore, char[] keyStorePassword, String alias) {
        try {
            return (PrivateKey)keyStore.getKey(alias, keyStorePassword);
        }
        catch (KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException e) {
            throw new RuntimeException(e);
        }
    }

    public JsonObject decode(String token) {
        String signatureSeg;
        String[] segments = token.split("\\.");
        if (segments.length != (this.unsecure ? 2 : 3)) {
            throw new RuntimeException("Not enough or too many segments");
        }
        String headerSeg = segments[0];
        String payloadSeg = segments[1];
        String string = signatureSeg = this.unsecure ? null : segments[2];
        if ("".equals(signatureSeg)) {
            throw new RuntimeException("Signature is required");
        }
        JsonObject header = new JsonObject(new String(JWT.base64urlDecode(headerSeg), UTF8));
        JsonObject payload = new JsonObject(new String(JWT.base64urlDecode(payloadSeg), UTF8));
        String alg = header.getString("alg");
        List<Crypto> cryptos = this.cryptoMap.get(alg);
        if (cryptos == null || cryptos.size() == 0) {
            throw new RuntimeException("Algorithm not supported");
        }
        if (!this.unsecure && "none".equals(alg)) {
            throw new RuntimeException("Algorithm \"none\" not allowed");
        }
        if (!this.unsecure) {
            byte[] payloadInput = JWT.base64urlDecode(signatureSeg);
            byte[] signingInput = (headerSeg + "." + payloadSeg).getBytes(UTF8);
            for (Crypto c : cryptos) {
                if (!c.verify(payloadInput, signingInput)) continue;
                return payload;
            }
            throw new RuntimeException("Signature verification failed");
        }
        return payload;
    }

    public String sign(JsonObject payload, JsonObject options) {
        Long expiresInSeconds;
        String algorithm = options.getString("algorithm", "HS256");
        List<Crypto> cryptos = this.cryptoMap.get(algorithm);
        if (cryptos == null || cryptos.size() == 0) {
            throw new RuntimeException("Algorithm not supported");
        }
        JsonObject header = new JsonObject().mergeIn(options.getJsonObject("header", EMPTY)).put("typ", "JWT").put("alg", algorithm);
        long timestamp = System.currentTimeMillis() / 1000L;
        if (!options.getBoolean("noTimestamp", Boolean.valueOf(false)).booleanValue()) {
            payload.put("iat", payload.getValue("iat", (Object)timestamp));
        }
        if ((expiresInSeconds = options.containsKey("expiresInMinutes") ? Long.valueOf(options.getLong("expiresInMinutes") * 60L) : options.getLong("expiresInSeconds")) != null) {
            payload.put("exp", Long.valueOf(timestamp + expiresInSeconds));
        }
        if (options.containsKey("audience")) {
            payload.put("aud", options.getValue("audience"));
        }
        if (options.containsKey("issuer")) {
            payload.put("iss", options.getValue("issuer"));
        }
        if (options.containsKey("subject")) {
            payload.put("sub", options.getValue("subject"));
        }
        String headerSegment = JWT.base64urlEncode(header.encode());
        String payloadSegment = JWT.base64urlEncode(payload.encode());
        String signingInput = headerSegment + "." + payloadSegment;
        String signSegment = JWT.base64urlEncode(cryptos.get(RND.nextInt(cryptos.size())).sign(signingInput.getBytes(UTF8)));
        return headerSegment + "." + payloadSegment + "." + signSegment;
    }

    private static byte[] base64urlDecode(String str) {
        return decoder.decode(str.getBytes(UTF8));
    }

    private static String base64urlEncode(String str) {
        return JWT.base64urlEncode(str.getBytes(UTF8));
    }

    private static String base64urlEncode(byte[] bytes) {
        return encoder.encodeToString(bytes);
    }

    public boolean isUnsecure() {
        return this.unsecure;
    }

    public Collection<String> availableAlgorithms() {
        return this.cryptoMap.keySet();
    }
}

