/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.org.apache.hadoop.hbase.backup;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.io.MultipleIOException;
import org.apache.hudi.org.apache.hadoop.hbase.backup.FailedArchiveException;
import org.apache.hudi.org.apache.hadoop.hbase.client.RegionInfo;
import org.apache.hudi.org.apache.hadoop.hbase.regionserver.HStoreFile;
import org.apache.hudi.org.apache.hadoop.hbase.util.Bytes;
import org.apache.hudi.org.apache.hadoop.hbase.util.CommonFSUtils;
import org.apache.hudi.org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hudi.org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hudi.org.apache.hadoop.hbase.util.HFileArchiveUtil;
import org.apache.hudi.org.apache.hadoop.hbase.util.Threads;
import org.apache.hudi.org.apache.hbase.thirdparty.com.google.common.base.Preconditions;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class HFileArchiver {
    private static final Logger LOG = LoggerFactory.getLogger(HFileArchiver.class);
    private static final String SEPARATOR = ".";
    private static final int DEFAULT_RETRIES_NUMBER = 3;
    private static final Function<File, Path> FUNC_FILE_TO_PATH = new Function<File, Path>(){

        @Override
        public Path apply(File file) {
            return file == null ? null : file.getPath();
        }
    };
    private static ThreadPoolExecutor archiveExecutor;

    private HFileArchiver() {
    }

    public static boolean exists(Configuration conf, FileSystem fs, RegionInfo info) throws IOException {
        Path rootDir = CommonFSUtils.getRootDir(conf);
        Path regionDir = FSUtils.getRegionDirFromRootDir(rootDir, info);
        return fs.exists(regionDir);
    }

    public static void archiveRegion(Configuration conf, FileSystem fs, RegionInfo info) throws IOException {
        Path rootDir = CommonFSUtils.getRootDir(conf);
        HFileArchiver.archiveRegion(fs, rootDir, CommonFSUtils.getTableDir(rootDir, info.getTable()), FSUtils.getRegionDirFromRootDir(rootDir, info));
    }

    public static boolean archiveRegion(FileSystem fs, Path rootdir, Path tableDir, Path regionDir) throws IOException {
        if (tableDir == null || regionDir == null) {
            LOG.error("No archive directory could be found because tabledir (" + tableDir + ") or regiondir (" + regionDir + "was null. Deleting files instead.");
            if (regionDir != null) {
                HFileArchiver.deleteRegionWithoutArchiving(fs, regionDir);
            }
            return false;
        }
        LOG.debug("ARCHIVING {}", (Object)regionDir);
        Preconditions.checkArgument(regionDir.toString().startsWith(tableDir.toString()));
        Path regionArchiveDir = HFileArchiveUtil.getRegionArchiveDir(rootdir, CommonFSUtils.getTableName(tableDir), regionDir.getName());
        FileStatusConverter getAsFile = new FileStatusConverter(fs);
        ArrayList<File> toArchive = new ArrayList<File>();
        final FSUtils.DirFilter dirFilter = new FSUtils.DirFilter(fs);
        PathFilter nonHidden = new PathFilter(){

            public boolean accept(Path file) {
                return dirFilter.accept(file) && !file.getName().startsWith(HFileArchiver.SEPARATOR);
            }
        };
        FileStatus[] storeDirs = CommonFSUtils.listStatus(fs, regionDir, nonHidden);
        if (storeDirs == null) {
            LOG.debug("Directory {} empty.", (Object)regionDir);
            return HFileArchiver.deleteRegionWithoutArchiving(fs, regionDir);
        }
        Stream.of(storeDirs).map(getAsFile).forEachOrdered(toArchive::add);
        LOG.debug("Archiving " + toArchive);
        List<File> failedArchive = HFileArchiver.resolveAndArchive(fs, regionArchiveDir, toArchive, EnvironmentEdgeManager.currentTime());
        if (!failedArchive.isEmpty()) {
            throw new FailedArchiveException("Failed to archive/delete all the files for region:" + regionDir.getName() + " into " + regionArchiveDir + ". Something is probably awry on the filesystem.", failedArchive.stream().map(FUNC_FILE_TO_PATH).collect(Collectors.toList()));
        }
        return HFileArchiver.deleteRegionWithoutArchiving(fs, regionDir);
    }

    public static void archiveRegions(Configuration conf, FileSystem fs, Path rootDir, Path tableDir, List<Path> regionDirList) throws IOException {
        ArrayList<Future<Void>> futures = new ArrayList<Future<Void>>(regionDirList.size());
        for (Path path : regionDirList) {
            Future<Void> future = HFileArchiver.getArchiveExecutor(conf).submit(() -> {
                HFileArchiver.archiveRegion(fs, rootDir, tableDir, regionDir);
                return null;
            });
            futures.add(future);
        }
        try {
            for (Future future : futures) {
                future.get();
            }
        }
        catch (InterruptedException e) {
            throw new InterruptedIOException(e.getMessage());
        }
        catch (ExecutionException e) {
            throw new IOException(e.getCause());
        }
    }

    private static synchronized ThreadPoolExecutor getArchiveExecutor(Configuration conf) {
        if (archiveExecutor == null) {
            int maxThreads = conf.getInt("hbase.hfilearchiver.thread.pool.max", 8);
            archiveExecutor = Threads.getBoundedCachedThreadPool(maxThreads, 30L, TimeUnit.SECONDS, HFileArchiver.getThreadFactory());
            Runtime.getRuntime().addShutdownHook(new Thread(() -> archiveExecutor.shutdown()));
        }
        return archiveExecutor;
    }

    private static ThreadFactory getThreadFactory() {
        return new ThreadFactory(){
            final AtomicInteger threadNumber = new AtomicInteger(1);

            @Override
            public Thread newThread(Runnable r) {
                String name = "HFileArchiver-" + this.threadNumber.getAndIncrement();
                Thread t = new Thread(r, name);
                t.setDaemon(true);
                return t;
            }
        };
    }

    public static void archiveFamily(FileSystem fs, Configuration conf, RegionInfo parent, Path tableDir, byte[] family) throws IOException {
        Path familyDir = new Path(tableDir, new Path(parent.getEncodedName(), Bytes.toString(family)));
        HFileArchiver.archiveFamilyByFamilyDir(fs, conf, parent, familyDir, family);
    }

    public static void archiveFamilyByFamilyDir(FileSystem fs, Configuration conf, RegionInfo parent, Path familyDir, byte[] family) throws IOException {
        FileStatus[] storeFiles = CommonFSUtils.listStatus(fs, familyDir);
        if (storeFiles == null) {
            LOG.debug("No files to dispose of in {}, family={}", (Object)parent.getRegionNameAsString(), (Object)Bytes.toString(family));
            return;
        }
        FileStatusConverter getAsFile = new FileStatusConverter(fs);
        Collection toArchive = Stream.of(storeFiles).map(getAsFile).collect(Collectors.toList());
        Path storeArchiveDir = HFileArchiveUtil.getStoreArchivePath(conf, parent, family);
        List<File> failedArchive = HFileArchiver.resolveAndArchive(fs, storeArchiveDir, toArchive, EnvironmentEdgeManager.currentTime());
        if (!failedArchive.isEmpty()) {
            throw new FailedArchiveException("Failed to archive/delete all the files for region:" + Bytes.toString(parent.getRegionName()) + ", family:" + Bytes.toString(family) + " into " + storeArchiveDir + ". Something is probably awry on the filesystem.", failedArchive.stream().map(FUNC_FILE_TO_PATH).collect(Collectors.toList()));
        }
    }

    public static void archiveStoreFiles(Configuration conf, FileSystem fs, RegionInfo regionInfo, Path tableDir, byte[] family, Collection<HStoreFile> compactedFiles) throws IOException {
        Path storeArchiveDir = HFileArchiveUtil.getStoreArchivePath(conf, regionInfo, tableDir, family);
        HFileArchiver.archive(fs, regionInfo, family, compactedFiles, storeArchiveDir);
    }

    public static void archiveRecoveredEdits(Configuration conf, FileSystem fs, RegionInfo regionInfo, byte[] family, Collection<HStoreFile> replayedEdits) throws IOException {
        String workingDir = conf.get("hbase.wal.dir", conf.get("hbase.rootdir"));
        Path path = new Path(workingDir);
        if (path.isAbsoluteAndSchemeAuthorityNull()) {
            path = new Path(conf.get("hbase.rootdir"));
        }
        if (path.toUri().getScheme() != null && !path.toUri().getScheme().equals(fs.getScheme())) {
            throw new IOException("Wrong file system! Should be " + path.toUri().getScheme() + ", but got " + fs.getScheme());
        }
        path = HFileArchiveUtil.getStoreArchivePathForRootDir(path, regionInfo, family);
        HFileArchiver.archive(fs, regionInfo, family, replayedEdits, path);
    }

    private static void archive(FileSystem fs, RegionInfo regionInfo, byte[] family, Collection<HStoreFile> compactedFiles, Path storeArchiveDir) throws IOException {
        if (fs == null) {
            LOG.warn("Passed filesystem is null, so just deleting files without archiving for {},family={}", (Object)Bytes.toString(regionInfo.getRegionName()), (Object)Bytes.toString(family));
            HFileArchiver.deleteStoreFilesWithoutArchiving(compactedFiles);
            return;
        }
        if (compactedFiles.isEmpty()) {
            LOG.debug("No files to dispose of, done!");
            return;
        }
        if (regionInfo == null || family == null) {
            throw new IOException("Need to have a region and a family to archive from.");
        }
        if (!fs.mkdirs(storeArchiveDir)) {
            throw new IOException("Could not make archive directory (" + storeArchiveDir + ") for store:" + Bytes.toString(family) + ", deleting compacted files instead.");
        }
        LOG.debug("Archiving compacted files.");
        StoreToFile getStorePath = new StoreToFile(fs);
        Collection storeFiles = compactedFiles.stream().map(getStorePath).collect(Collectors.toList());
        List<File> failedArchive = HFileArchiver.resolveAndArchive(fs, storeArchiveDir, storeFiles, EnvironmentEdgeManager.currentTime());
        if (!failedArchive.isEmpty()) {
            throw new FailedArchiveException("Failed to archive/delete all the files for region:" + Bytes.toString(regionInfo.getRegionName()) + ", family:" + Bytes.toString(family) + " into " + storeArchiveDir + ". Something is probably awry on the filesystem.", failedArchive.stream().map(FUNC_FILE_TO_PATH).collect(Collectors.toList()));
        }
    }

    public static void archiveStoreFile(Configuration conf, FileSystem fs, RegionInfo regionInfo, Path tableDir, byte[] family, Path storeFile) throws IOException {
        Path storeArchiveDir = HFileArchiveUtil.getStoreArchivePath(conf, regionInfo, tableDir, family);
        if (!fs.mkdirs(storeArchiveDir)) {
            throw new IOException("Could not make archive directory (" + storeArchiveDir + ") for store:" + Bytes.toString(family) + ", deleting compacted files instead.");
        }
        FileablePath file = new FileablePath(fs, storeFile);
        long start2 = EnvironmentEdgeManager.currentTime();
        if (!HFileArchiver.resolveAndArchiveFile(storeArchiveDir, file, Long.toString(start2))) {
            throw new IOException("Failed to archive/delete the file for region:" + regionInfo.getRegionNameAsString() + ", family:" + Bytes.toString(family) + " into " + storeArchiveDir + ". Something is probably awry on the filesystem.");
        }
    }

    private static List<File> resolveAndArchive(FileSystem fs, Path baseArchiveDir, Collection<File> toArchive, long start2) throws IOException {
        if (toArchive.isEmpty()) {
            return Collections.emptyList();
        }
        LOG.trace("Moving files to the archive directory {}", (Object)baseArchiveDir);
        if (!fs.exists(baseArchiveDir)) {
            if (!fs.mkdirs(baseArchiveDir)) {
                throw new IOException("Failed to create the archive directory:" + baseArchiveDir + ", quitting archive attempt.");
            }
            LOG.trace("Created archive directory {}", (Object)baseArchiveDir);
        }
        ArrayList<File> failures = new ArrayList<File>();
        String startTime = Long.toString(start2);
        for (File file : toArchive) {
            try {
                LOG.trace("Archiving {}", (Object)file);
                if (file.isFile()) {
                    if (HFileArchiver.resolveAndArchiveFile(baseArchiveDir, file, startTime)) continue;
                    LOG.warn("Couldn't archive " + file + " into backup directory: " + baseArchiveDir);
                    failures.add(file);
                    continue;
                }
                LOG.trace("{} is a directory, archiving children files", (Object)file);
                Path parentArchiveDir = new Path(baseArchiveDir, file.getName());
                Collection<File> children = file.getChildren();
                failures.addAll(HFileArchiver.resolveAndArchive(fs, parentArchiveDir, children, start2));
            }
            catch (IOException e) {
                LOG.warn("Failed to archive {}", (Object)file, (Object)e);
                failures.add(file);
            }
        }
        return failures;
    }

    private static boolean resolveAndArchiveFile(Path archiveDir, File currentFile, String archiveStartTime) throws IOException {
        String filename = currentFile.getName();
        Path archiveFile = new Path(archiveDir, filename);
        FileSystem fs = currentFile.getFileSystem();
        if (fs.exists(archiveFile)) {
            LOG.debug("{} already exists in archive, moving to timestamped backup and overwriting current.", (Object)archiveFile);
            Path backedupArchiveFile = new Path(archiveDir, filename + SEPARATOR + archiveStartTime);
            if (!fs.rename(archiveFile, backedupArchiveFile)) {
                LOG.error("Could not rename archive file to backup: " + backedupArchiveFile + ", deleting existing file in favor of newer.");
                if (!fs.delete(archiveFile, false)) {
                    throw new IOException("Couldn't delete existing archive file (" + archiveFile + ") or rename it to the backup file (" + backedupArchiveFile + ") to make room for similarly named file.");
                }
            }
            LOG.debug("Backed up archive file from " + archiveFile);
        }
        LOG.trace("No existing file in archive for {}, free to archive original file.", (Object)archiveFile);
        boolean success = false;
        for (int i = 0; !success && i < 3; ++i) {
            if (i > 0) {
                try {
                    if (!fs.exists(archiveDir) && fs.mkdirs(archiveDir)) {
                        LOG.debug("Created archive directory {}", (Object)archiveDir);
                    }
                }
                catch (IOException e) {
                    LOG.warn("Failed to create directory {}", (Object)archiveDir, (Object)e);
                }
            }
            try {
                success = currentFile.moveAndClose(archiveFile);
                continue;
            }
            catch (FileNotFoundException fnfe) {
                LOG.warn("Failed to archive " + currentFile + " because it does not exist! Skipping and continuing on.", (Throwable)fnfe);
                success = true;
                continue;
            }
            catch (IOException e) {
                LOG.warn("Failed to archive " + currentFile + " on try #" + i, (Throwable)e);
                success = false;
            }
        }
        if (!success) {
            LOG.error("Failed to archive " + currentFile);
            return false;
        }
        LOG.debug("Archived from {} to {}", (Object)currentFile, (Object)archiveFile);
        return true;
    }

    private static boolean deleteRegionWithoutArchiving(FileSystem fs, Path regionDir) throws IOException {
        if (fs.delete(regionDir, true)) {
            LOG.debug("Deleted {}", (Object)regionDir);
            return true;
        }
        LOG.debug("Failed to delete directory {}", (Object)regionDir);
        return false;
    }

    private static void deleteStoreFilesWithoutArchiving(Collection<HStoreFile> compactedFiles) throws IOException {
        LOG.debug("Deleting files without archiving.");
        ArrayList<IOException> errors2 = new ArrayList<IOException>(0);
        for (HStoreFile hsf : compactedFiles) {
            try {
                hsf.deleteStoreFile();
            }
            catch (IOException e) {
                LOG.error("Failed to delete {}", (Object)hsf.getPath());
                errors2.add(e);
            }
        }
        if (errors2.size() > 0) {
            throw MultipleIOException.createIOException(errors2);
        }
    }

    private static class FileableStoreFile
    extends File {
        HStoreFile file;

        public FileableStoreFile(FileSystem fs, HStoreFile store) {
            super(fs);
            this.file = store;
        }

        @Override
        public void delete() throws IOException {
            this.file.deleteStoreFile();
        }

        @Override
        public String getName() {
            return this.file.getPath().getName();
        }

        @Override
        public boolean isFile() {
            return true;
        }

        @Override
        public Collection<File> getChildren() throws IOException {
            return Collections.emptyList();
        }

        @Override
        public void close() throws IOException {
            this.file.closeStoreFile(true);
        }

        @Override
        Path getPath() {
            return this.file.getPath();
        }
    }

    private static class FileablePath
    extends File {
        private final Path file;
        private final FileStatusConverter getAsFile;

        public FileablePath(FileSystem fs, Path file) {
            super(fs);
            this.file = file;
            this.getAsFile = new FileStatusConverter(fs);
        }

        @Override
        public void delete() throws IOException {
            if (!this.fs.delete(this.file, true)) {
                throw new IOException("Failed to delete:" + this.file);
            }
        }

        @Override
        public String getName() {
            return this.file.getName();
        }

        @Override
        public Collection<File> getChildren() throws IOException {
            if (this.fs.isFile(this.file)) {
                return Collections.emptyList();
            }
            return Stream.of(this.fs.listStatus(this.file)).map(this.getAsFile).collect(Collectors.toList());
        }

        @Override
        public boolean isFile() throws IOException {
            return this.fs.isFile(this.file);
        }

        @Override
        public void close() throws IOException {
        }

        @Override
        Path getPath() {
            return this.file;
        }
    }

    private static abstract class File {
        protected final FileSystem fs;

        public File(FileSystem fs) {
            this.fs = fs;
        }

        abstract void delete() throws IOException;

        abstract boolean isFile() throws IOException;

        abstract Collection<File> getChildren() throws IOException;

        abstract void close() throws IOException;

        abstract String getName();

        abstract Path getPath();

        public boolean moveAndClose(Path dest) throws IOException {
            this.close();
            Path p = this.getPath();
            return CommonFSUtils.renameAndSetModifyTime(this.fs, p, dest);
        }

        public FileSystem getFileSystem() {
            return this.fs;
        }

        public String toString() {
            return this.getClass().getSimpleName() + ", " + this.getPath().toString();
        }
    }

    private static class StoreToFile
    extends FileConverter<HStoreFile> {
        public StoreToFile(FileSystem fs) {
            super(fs);
        }

        @Override
        public File apply(HStoreFile input) {
            return new FileableStoreFile(this.fs, input);
        }
    }

    private static class FileStatusConverter
    extends FileConverter<FileStatus> {
        public FileStatusConverter(FileSystem fs) {
            super(fs);
        }

        @Override
        public File apply(FileStatus input) {
            return new FileablePath(this.fs, input.getPath());
        }
    }

    private static abstract class FileConverter<T>
    implements Function<T, File> {
        protected final FileSystem fs;

        public FileConverter(FileSystem fs) {
            this.fs = fs;
        }
    }
}

