package com.aliyun.dashvector;

import com.aliyun.dashvector.common.Constants;
import com.aliyun.dashvector.common.ErrorCode;
import com.aliyun.dashvector.models.*;
import com.aliyun.dashvector.models.Doc;
import com.aliyun.dashvector.models.DocOpResult;
import com.aliyun.dashvector.models.PartitionStats;
import com.aliyun.dashvector.models.requests.DeleteDocRequest;
import com.aliyun.dashvector.models.requests.FetchDocRequest;
import com.aliyun.dashvector.models.requests.InsertDocRequest;
import com.aliyun.dashvector.models.requests.QueryDocGroupByRequest;
import com.aliyun.dashvector.models.requests.QueryDocRequest;
import com.aliyun.dashvector.models.requests.UpdateDocRequest;
import com.aliyun.dashvector.models.requests.UpsertDocRequest;
import com.aliyun.dashvector.models.responses.Response;
import com.aliyun.dashvector.proto.*;
import com.aliyun.dashvector.utils.Validator;
import com.aliyun.dashvector.validator.ValidatorFactory;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.MoreExecutors;
import io.grpc.Metadata;
import io.grpc.stub.MetadataUtils;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import lombok.Getter;
import lombok.NonNull;

/**
 * @author hq90172
 */
public class DashVectorCollection {
  @Getter private String name;
  @Getter private CollectionMeta collectionMeta;
  @Getter private final int code;
  @Getter private final String message;
  @Getter private final String requestId;
  private DashVectorServiceGrpc.DashVectorServiceFutureStub stub;

  public DashVectorCollection(
      Response<CollectionMeta> response, DashVectorServiceGrpc.DashVectorServiceFutureStub stub) {
    this.code = response.getCode();
    this.message = response.getMessage();
    this.requestId = response.getRequestId();
    if (response.isSuccess()) {
      Metadata metadata = new Metadata();
      CollectionMeta collectionMeta = response.getOutput();
      metadata.put(
          Metadata.Key.of(Constants.HEADER_COLLECTION_NAME, Metadata.ASCII_STRING_MARSHALLER),
          collectionMeta.getName());
      this.name = collectionMeta.getName();
      this.stub = stub.withInterceptors(MetadataUtils.newAttachHeadersInterceptor(metadata));
      this.collectionMeta = collectionMeta;
    }
  }

  public Boolean isSuccess() {
    return code == ErrorCode.SUCCESS.getCode();
  }

  public Response<List<DocOpResult>> insert(@NonNull InsertDocRequest insertDocRequest) {
    if (!isSuccess()) {
      return Response.create(code, message, requestId, null);
    }
    try {
      return this.insertAsync(insertDocRequest).get();
    } catch (Exception e) {
      return Response.failed(e);
    }
  }

  public ListenableFuture<Response<List<DocOpResult>>> insertAsync(
      @NonNull InsertDocRequest insertDocRequest) {
    if (!isSuccess()) {
      return Futures.immediateFuture(Response.create(code, message, requestId, null));
    }
    try {
      ListenableFuture<InsertDocResponse> response =
          this.stub.insertDoc(insertDocRequest.toProto(this.collectionMeta,
                  com.aliyun.dashvector.proto.DocOpResult.DocOp.insert));
      return Futures.transform(response, Response::success, MoreExecutors.directExecutor());
    } catch (Exception e) {
      return Futures.immediateFuture(Response.failed(e));
    }
  }

  public Response<List<DocOpResult>> upsert(@NonNull UpsertDocRequest upsertDocRequest) {
    if (!isSuccess()) {
      return Response.create(code, message, requestId, null);
    }
    try {
      return this.upsertAsync(upsertDocRequest).get();
    } catch (Exception e) {
      return Response.failed(e);
    }
  }

  public ListenableFuture<Response<List<DocOpResult>>> upsertAsync(
      @NonNull UpsertDocRequest upsertDocRequest) {
    if (!isSuccess()) {
      return Futures.immediateFuture(Response.create(code, message, requestId, null));
    }
    try {
      ListenableFuture<UpsertDocResponse> response =
          this.stub.upsertDoc(upsertDocRequest.toProto(this.collectionMeta,
                  com.aliyun.dashvector.proto.DocOpResult.DocOp.upsert));
      return Futures.transform(response, Response::success, MoreExecutors.directExecutor());
    } catch (Exception e) {
      return Futures.immediateFuture(Response.failed(e));
    }
  }

  public Response<List<DocOpResult>> update(@NonNull UpdateDocRequest updateDocRequest) {
    if (!isSuccess()) {
      return Response.create(code, message, requestId, null);
    }
    try {
      return this.updateAsync(updateDocRequest).get();
    } catch (Exception e) {
      return Response.failed(e);
    }
  }

  public ListenableFuture<Response<List<DocOpResult>>> updateAsync(
      @NonNull UpdateDocRequest updateDocRequest) {
    if (!isSuccess()) {
      return Futures.immediateFuture(Response.create(code, message, requestId, null));
    }
    try {
      ListenableFuture<UpdateDocResponse> response =
          this.stub.updateDoc(updateDocRequest.toProto(this.collectionMeta,
                  com.aliyun.dashvector.proto.DocOpResult.DocOp.update));
      return Futures.transform(response, Response::success, MoreExecutors.directExecutor());
    } catch (Exception e) {
      return Futures.immediateFuture(Response.failed(e));
    }
  }

  public Response<List<com.aliyun.dashvector.models.Doc>> query(
      @NonNull QueryDocRequest queryDocRequest) {
    if (!isSuccess()) {
      return Response.create(code, message, requestId, null);
    }
    try {
      return this.queryAsync(queryDocRequest).get();
    } catch (Exception e) {
      return Response.failed(e);
    }
  }

  public ListenableFuture<Response<List<Doc>>> queryAsync(
      @NonNull QueryDocRequest queryDocRequest) {
    if (!isSuccess()) {
      return Futures.immediateFuture(Response.create(code, message, requestId, null));
    }
    try {
      ValidatorFactory.create(collectionMeta).validateQueryDocRequest(queryDocRequest);
      ListenableFuture<QueryDocResponse> future =
          this.stub.queryDoc(queryDocRequest.toProto(this.collectionMeta));
      Function<QueryDocResponse, Response<List<Doc>>> transformFunc =
          queryDocResponse -> Response.success(queryDocResponse, this.collectionMeta);
      return Futures.transform(future, transformFunc::apply, MoreExecutors.directExecutor());
    } catch (Exception e) {
      return Futures.immediateFuture(Response.failed(e));
    }
  }

  public Response<List<com.aliyun.dashvector.models.Group>> queryGroupBy(
      @NonNull QueryDocGroupByRequest queryDocGroupByRequest) {
    if (!isSuccess()) {
      return Response.create(code, message, requestId, null);
    }
    try {
      return this.queryGroupByAsync(queryDocGroupByRequest).get();
    } catch (Exception e) {
      return Response.failed(e);
    }
  }

  public ListenableFuture<Response<List<Group>>> queryGroupByAsync(
      @NonNull QueryDocGroupByRequest queryDocGroupByRequest) {
    if (!isSuccess()) {
      return Futures.immediateFuture(Response.create(code, message, requestId, null));
    }
    try {
      ValidatorFactory.create(collectionMeta).validateQueryDocGroupByRequest(queryDocGroupByRequest);
      ListenableFuture<QueryDocGroupByResponse> future =
          this.stub.queryDocGroupBy(
              queryDocGroupByRequest.toProto(this.collectionMeta));
      Function<QueryDocGroupByResponse, Response<List<Group>>> transformFunc =
          queryDocResponse -> Response.success(queryDocResponse, this.collectionMeta);
      return Futures.transform(future, transformFunc::apply, MoreExecutors.directExecutor());
    } catch (Exception e) {
      return Futures.immediateFuture(Response.failed(e));
    }
  }

  public Response<List<DocOpResult>> delete(@NonNull DeleteDocRequest deleteDocRequest) {
    if (!isSuccess()) {
      return Response.create(code, message, requestId, null);
    }
    try {
      return this.deleteAsync(deleteDocRequest).get();
    } catch (Exception e) {
      return Response.failed(e);
    }
  }

  public ListenableFuture<Response<List<DocOpResult>>> deleteAsync(
      @NonNull DeleteDocRequest deleteDocRequest) {
    if (!isSuccess()) {
      return Futures.immediateFuture(Response.create(code, message, requestId, null));
    }
    try {
      ListenableFuture<DeleteDocResponse> future = this.stub.deleteDoc(deleteDocRequest.toProto());
      return Futures.transform(future, Response::success, MoreExecutors.directExecutor());
    } catch (Exception e) {
      return Futures.immediateFuture(Response.failed(e));
    }
  }

  public Response<Map<String, Doc>> fetch(@NonNull FetchDocRequest fetchDocRequest) {
    if (!isSuccess()) {
      return Response.create(code, message, requestId, null);
    }
    try {
      return this.fetchAsync(fetchDocRequest).get();
    } catch (Exception e) {
      return Response.failed(e);
    }
  }

  public ListenableFuture<Response<Map<String, Doc>>> fetchAsync(
      @NonNull FetchDocRequest fetchDocRequest) {
    if (!isSuccess()) {
      return Futures.immediateFuture(Response.create(code, message, requestId, null));
    }
    try {
      ListenableFuture<FetchDocResponse> future = this.stub.fetchDoc(fetchDocRequest.toProto());
      Function<FetchDocResponse, Response<Map<String, Doc>>> transformFunc =
          fetchDocResponse -> Response.success(fetchDocResponse, this.collectionMeta);
      return Futures.transform(future, transformFunc::apply, MoreExecutors.directExecutor());
    } catch (Exception e) {
      return Futures.immediateFuture(Response.failed(e));
    }
  }

  public Response<CollectionStats> stats() {
    if (!isSuccess()) {
      return Response.create(code, message, requestId, null);
    }
    try {
      StatsCollectionResponse response =
          this.stub.statsCollection(StatsCollectionRequest.getDefaultInstance()).get();
      return Response.success(response);
    } catch (Exception e) {
      return Response.failed(e);
    }
  }

  public Response<Void> createPartition(@NonNull String name) {
    return this.createPartition(name, null);
  }

  public Response<Void> createPartition(@NonNull String name, Integer timeout) {
    if (!isSuccess()) {
      return Response.create(code, message, requestId, null);
    }
    CreatePartitionResponse response;
    try {
      Validator.verifyPartitionName(name);
      CreatePartitionRequest request = CreatePartitionRequest.newBuilder().setName(name).build();
      response = this.stub.createPartition(request).get();

      if (response.getCode() != ErrorCode.SUCCESS.getCode() && Objects.isNull(timeout)) {
        return Response.create(
            response.getCode(), response.getMessage(), response.getRequestId(), null);
      }
    } catch (Exception e) {
      return Response.failed(e);
    }

    int createTimeout = Objects.isNull(timeout) ? 0 : timeout;
    int retryCount = 0;
    while (true) {
      Response<Status> describeResponse = this.describePartition(name);
      if (describeResponse.getCode() == ErrorCode.SUCCESS.getCode()) {
        String status = describeResponse.getOutput().name();
        if ("ERROR".equals(status) || "DROPPING".equals(status)) {
          return Response.create(
              ErrorCode.UNREADY_PARTITION.getCode(),
              String.format(
                  "DashVectorSDK collection(%s)'s partition(%s) unready. Status is %s ",
                  collectionMeta.getName(), name, status),
              describeResponse.getRequestId(),
              null);
        } else if ("SERVING".equals(status)) {
          return Response.success(
              response.getCode(), response.getMessage(), response.getRequestId());
        }
      } else {
        retryCount++;
      }
      if (retryCount > 3) {
        return Response.create(
            describeResponse.getCode(),
            String.format("DashVectorSDK Get Partition(%s) Status create", name),
            null,
            null);
      }
      try {
        Thread.sleep(5000);
      } catch (Exception e) {
        return Response.failed(e);
      }

      if (Objects.isNull(timeout)) {
        continue;
      }

      createTimeout -= 5;
      if (createTimeout < 0) {
        return Response.create(
            ErrorCode.TIMEOUT.getCode(),
            String.format("DashVectorSDK Get Partition(%s) Status Timeout.", name),
            null,
            null);
      }
    }
  }

  public Response<Status> describePartition(@NonNull String name) {
    if (!isSuccess()) {
      return Response.create(code, message, requestId, null);
    }
    try {
      Validator.verifyPartitionName(name);
      DescribePartitionRequest req = DescribePartitionRequest.newBuilder().setName(name).build();
      DescribePartitionResponse response = this.stub.describePartition(req).get();

      return Response.success(response);
    } catch (Exception e) {
      return Response.failed(e);
    }
  }

  public Response<List<String>> listPartitions() {
    if (!isSuccess()) {
      return Response.create(code, message, requestId, null);
    }
    try {
      ListPartitionsResponse response =
          this.stub.listPartitions(ListPartitionsRequest.getDefaultInstance()).get();
      return Response.success(response);
    } catch (Exception e) {
      return Response.failed(e);
    }
  }

  public Response<PartitionStats> statsPartition(@NonNull String name) {
    if (!isSuccess()) {
      return Response.create(code, message, requestId, null);
    }
    try {
      Validator.verifyPartitionName(name);
      StatsPartitionRequest req = StatsPartitionRequest.newBuilder().setName(name).build();
      StatsPartitionResponse response = this.stub.statsPartition(req).get();

      return Response.success(response);
    } catch (Exception e) {
      return Response.failed(e);
    }
  }

  public Response<Void> deletePartition(@NonNull String name) {
    if (!isSuccess()) {
      return Response.create(code, message, requestId, null);
    }
    try {
      Validator.verifyPartitionName(name);
      DeletePartitionRequest req = DeletePartitionRequest.newBuilder().setName(name).build();
      DeletePartitionResponse response = this.stub.deletePartition(req).get();

      return Response.success(response);
    } catch (Exception e) {
      return Response.failed(e);
    }
  }
}
