package com.aliyun.dashvector.models.requests;

import com.aliyun.dashvector.common.Constants;
import com.aliyun.dashvector.common.DashVectorException;
import com.aliyun.dashvector.models.*;
import com.aliyun.dashvector.models.Vector;
import com.aliyun.dashvector.utils.Convertor;
import com.aliyun.dashvector.utils.Validator;

import java.util.*;

import com.aliyun.dashvector.validator.ValidatorFactory;
import lombok.Builder;
import lombok.Getter;
import lombok.NonNull;
import lombok.Singular;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;

/**
 * @author sanyi
 */
@Getter
@Builder
public class QueryDocRequest {
    private Vector vector;
    private VectorQuery vectorQuery;
    private String id;
    @Builder.Default
    private int topk = 10;
    @Builder.Default
    private boolean includeVector = false;
    @Builder.Default
    @NonNull
    private String partition = Constants.DEFAULT_PARTITION_NAME;
    private String filter;
    @Singular
    private List<String> outputFields;
    private Map<Long, Float> sparseVector;
    @Singular("vectors")
    private Map<String, VectorQuery> vectors;
    @Singular("sparseVectors")
    private Map<String, SparseVectorQuery> sparseVectors;
    private Ranker ranker;

    public com.aliyun.dashvector.proto.QueryDocRequest toProto(CollectionMeta collectionMeta)
            throws DashVectorException {
        // validate
        ValidatorFactory.create(collectionMeta).validateQueryDocRequest(this);

        // convert to protobuf
        com.aliyun.dashvector.proto.QueryDocRequest.Builder builder =
                com.aliyun.dashvector.proto.QueryDocRequest.newBuilder()
                        .setTopk(topk)
                        .setPartition(partition)
                        .setIncludeVector(includeVector);
        if (StringUtils.isNotEmpty(id)) {
            // convert id
            builder.setId(id);
        } else {
            // convert dense vector
            if (MapUtils.isNotEmpty(vectors)) {
                vectors.forEach((k, v) -> builder.putVectors(k, Convertor.toVectorQuery(v, collectionMeta.getDataType(k))));
            } else if (Objects.nonNull(vectorQuery)) {
                builder.putVectors("", Convertor.toVectorQuery(vectorQuery, collectionMeta.getDataType()));
            } else if (Objects.nonNull(vector)) {
                builder.setVector(Convertor.toVector(vector.getValue(), collectionMeta.getDataType()));
            }

            // convert sparse vector
            if (MapUtils.isNotEmpty(sparseVectors)) {
                sparseVectors.forEach((k, v) -> builder.putSparseVectors(k, Convertor.toSparseVectorQuery(v,
                        collectionMeta.getDataType(k))));
            } else if (Objects.nonNull(sparseVector)) {
                builder.putAllSparseVector(Convertor.toDashVectorSparse(sparseVector));
            }
        }

        // convert output fields
        if (CollectionUtils.isNotEmpty(outputFields)) {
            builder.addAllOutputFields(outputFields);
        }
        // convert filter
        if (StringUtils.isNotEmpty(filter)) {
            builder.setFilter(filter);
        }
        // convert rerank
        if (Objects.nonNull(ranker)) {
            builder.setRerank(ranker.toProto());
        }
        com.aliyun.dashvector.proto.QueryDocRequest request = builder.build();
        Validator.verifyRequestSize(request);
        return request;
    }
}
