package com.aliyun.dashvector.utils;

import com.aliyun.dashvector.common.Constants;
import com.aliyun.dashvector.common.DashVectorException;
import com.aliyun.dashvector.common.ErrorCode;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.lang.management.OperatingSystemMXBean;
import java.util.Objects;
import java.util.Properties;

/**
 * @author sanyi
 */
public class Utils {
  private static final ObjectMapper MAPPER = new ObjectMapper();

  static {
    MAPPER.setSerializationInclusion(JsonInclude.Include.NON_EMPTY);
  }

  /**
   * Json Parse
   *
   * @param o {@link Object}
   * @return the string of json
   * @throws DashVectorException When parse json to string failed
   */
  public static String toString(Object o) throws DashVectorException {
    try {
      return MAPPER.writeValueAsString(o);
    } catch (JsonProcessingException e) {
      throw new DashVectorException(
          ErrorCode.INVALID_JSON.getCode(),
          String.format("Json serialization failed. exception_msg:%s, object_type:%s",
                  e.getMessage(),
                  o.getClass().getName()));
    }
  }

  /**
   * Get insecure mode to start grpc with tls
   *
   * @return {@code true} if set DASHVECTOR_INSECURE_MODE=1
   */
  public static boolean isInSecureMode() {
    String inSecureMode = System.getenv(Constants.ENV_MODE_INSECURE);
    if (Objects.isNull(inSecureMode) || inSecureMode.isEmpty()) {
      return false;
    }
    return "1".equalsIgnoreCase(inSecureMode) || "true".equalsIgnoreCase(inSecureMode);
  }

  /**
   * Get sdk version form properties
   *
   * @return the version of dashvector-java-sdk
   */
  public static String getSdkVersion() {
    String version = null;
    Properties properties = new Properties();
    try {
      properties.load(Utils.class.getClassLoader().getResourceAsStream("dashvector.properties"));
      if (!properties.isEmpty()) {
        version = properties.getProperty("dashvector.version");
      }
    } catch (IOException e) {
      throw new DashVectorException(e, ErrorCode.UNKNOWN.getCode());
    }
    return version;
  }

  /**
   * Get user-agent. Contains sdk/grpc version and platform
   *
   * @return the useragent
   */
  public static String getUserAgent() {
    OperatingSystemMXBean osBean = ManagementFactory.getOperatingSystemMXBean();
    return String.format(
        "%s;%s;%s_%s_%s",
        getSdkVersion(),
        System.getProperty("java.version"),
        osBean.getName(),
        osBean.getArch(),
        osBean.getVersion());
  }
}
