package com.bxm.acl.facade;

import com.bxm.acl.facade.model.ResultModel;
import com.bxm.acl.facade.model.UserVo;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.function.Function;

/**
 * @author yirenjie
 * createDate:  2018/8/6
 */
public class AclAuthInterceptor implements HandlerInterceptor {
    private static final Logger LOGGER = LoggerFactory.getLogger(AclAuthInterceptor.class);
    private final int SYSTEM_ID;   // 系统id
    public static final String USER_KEY = "user";
    protected static final String SESSION_USER_NAME = "user_name";

    @Resource
    private AclUserManagerService aclUserManagerService;
    private Function<String, String> changeRequestURI;
    /**
     * 当未登录或没有权限时，是否抛出异常{@link NoPermissionException}，默认false
     * true: 抛出异常，false: 不抛出异常
     */
    private boolean throwException;

    public AclAuthInterceptor(int systemId) {
        this(systemId, null);
    }

    public AclAuthInterceptor(int systemId, Function<String, String> changeRequestURI) {
        this(systemId, changeRequestURI, false);
    }

    public AclAuthInterceptor(int systemId, Function<String, String> changeRequestURI, boolean throwException) {
        this.SYSTEM_ID = systemId;
        this.changeRequestURI = changeRequestURI;
        this.throwException = throwException;
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object o) throws Exception {
        String token = Utils.getToken(request);
        try {
            LOGGER.debug("请求权限管理系统,token={},systemId={},uri={}", token, this.SYSTEM_ID, request.getRequestURI());
            if (token == null) {
                if (throwException) {
                    throw new NoPermissionException("用户未登录");
                }
                Utils.sendErrorMsg(response, "用户未登录", "10000002");
                return false;
            }
            String checkUri = request.getRequestURI();
            if (null != changeRequestURI) {
                checkUri = changeRequestURI.apply(checkUri);
            }
            String ip = Utils.getIp(request);
            ResultModel<UserVo> result = aclUserManagerService.auth(token, this.SYSTEM_ID, checkUri);
            if (!result.isSuccessed()) {
                if (throwException) {
                    throw new NoPermissionException(result.getErrorDesc());
                }
                Utils.sendErrorMsg(response, result.getErrorDesc(), result.getErrorCode());
                LOGGER.warn("权限系统拒绝,msg={}", result.getErrorDesc());
                return false;
            }
            if (StringUtils.equals(HttpStatus.FORBIDDEN.value() + "", result.getErrorCode())) {
                if (throwException) {
                    throw new NoPermissionException("权限不足");
                }
                Utils.sendErrorMsg(response, "权限不足,如需继续访问请联系管理员", "");
                LOGGER.info("{}({}) - SYSTEM: {} 无权限，访问路径：{} 鉴权路径：{}", token, ip, this.SYSTEM_ID, request.getRequestURI(), checkUri);
                return false;
            }
            UserVo userVo = result.getReturnValue();
            request.setAttribute(USER_KEY, userVo);
            request.setAttribute(SESSION_USER_NAME, userVo.getUsername());

            LOGGER.info("{}({}) - 用户权限验证成功，访问路径：{} 鉴权路径：{}", userVo.getUsername(), ip, request.getRequestURI(), checkUri);
        } catch (Exception e) {
            if (throwException) {
                if (e instanceof NoPermissionException) {
                    throw e;
                }
                throw new NoPermissionException("权限系统异常,请稍候再试", e);
            }
            LOGGER.error("请求权限管理系统发生异常,token={},systemId={},uri={}:", token, this.SYSTEM_ID, request.getRequestURI(), e);
            Utils.sendErrorMsg(response, "权限系统异常,请稍候再试", "");
            return false;
        }
        return true;
    }

    @Override
    public void postHandle(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse, Object o, ModelAndView modelAndView) throws Exception {

    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object o, Exception e) throws Exception {

    }

    public static class NoPermissionException extends RuntimeException {

        public NoPermissionException() {
        }

        public NoPermissionException(String message) {
            super(message);
        }

        public NoPermissionException(String message, Throwable cause) {
            super(message, cause);
        }

        public NoPermissionException(Throwable cause) {
            super(cause);
        }

        public NoPermissionException(String message, Throwable cause, boolean enableSuppression, boolean writableStackTrace) {
            super(message, cause, enableSuppression, writableStackTrace);
        }
    }
}
