package com.bxm.report.service.datapark.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.datapark.facade.Page;
import com.bxm.report.facade.constant.IntegrationConstant;
import com.bxm.report.facade.datapark.DataparkTicketAnalysisIntegration;
import com.bxm.report.model.dto.analysis.AnalysisParamsDTO;
import com.bxm.report.service.datapark.TicketAnalysisService;
import com.bxm.report.service.utils.TicketAnalysisHelper;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.util.Collections;
import java.util.Map;

/**
 * 文 件 名 :
 * CopyRright (c) 2008-xxxx:
 * 文件编号：
 * 创 建 人： hxpeng
 * 日 期： 2018/8/15 10:52
 * 修 改 人：
 * 日 期：
 * 描 述： 券分析接口实现类
 * 版 本 号： v1
 */
@Slf4j
@Service
public class TicketAnalysisServiceImpl implements TicketAnalysisService {

    @Autowired
    private DataparkTicketAnalysisIntegration analysisIntegration;

    @Autowired
    private TicketAnalysisHelper ticketAnalysisHelper;

    @Override
    public Page analysis(AnalysisParamsDTO paramsDTO, String analysisType) {
        com.bxm.datapark.facade.ticket.model.dto.AnalysisParamsDTO analysisParamsDTO = this.convert(paramsDTO);
        switch (analysisType) {
            case IntegrationConstant.Analysis.ANALYSIS_TYPE_TREND:
                return ticketAnalysisHelper.convertAndBuildByTrend(analysisIntegration.trend(analysisParamsDTO));
            case IntegrationConstant.Analysis.ANALYSIS_TYPE_MEDIA:
                if (StringUtils.isBlank(paramsDTO.getMediaGroupType())) {
                    paramsDTO.setMediaGroupType(IntegrationConstant.Analysis.MEDIA_DEFAULT_FIELD_NAME);
                }
                return ticketAnalysisHelper.convertAndBuildByMedia(analysisIntegration.media(analysisParamsDTO));
            case IntegrationConstant.Analysis.ANALYSIS_TYPE_TAG_TYPE:
                return ticketAnalysisHelper.convertAndBuildByTagType(analysisIntegration.tagType(analysisParamsDTO));
            case IntegrationConstant.Analysis.ANALYSIS_TYPE_REGION:
                if (StringUtils.isBlank(paramsDTO.getRegionGroupType())) {
                    paramsDTO.setRegionGroupType(IntegrationConstant.Analysis.REGION_DEFAULT_FIELD_NAME);
                }
                return ticketAnalysisHelper.convertAndBuildByRegion(analysisIntegration.region(analysisParamsDTO));
            case IntegrationConstant.Analysis.ANALYSIS_TYPE_APP_OS:
                return ticketAnalysisHelper.convertAndBuildByAppOS(analysisIntegration.appOS(analysisParamsDTO));
            case IntegrationConstant.Analysis.ANALYSIS_TYPE_PAYMENT_TYPE:
                return ticketAnalysisHelper.convertAndBuildByPaymentType(analysisIntegration.paymentType(analysisParamsDTO));
            default:
                return null;
        }
    }

    @Override
    @SuppressWarnings("unchecked")
    public Map analysisChart(AnalysisParamsDTO paramsDTO, String analysisType) {
        com.bxm.datapark.facade.ticket.model.dto.AnalysisParamsDTO analysisParamsDTO = this.convert(paramsDTO);
        Map<String, Object> map = analysisIntegration.commonAnalysisChat(analysisParamsDTO, analysisType);
        if (MapUtils.isEmpty(map)) {
            return Collections.emptyMap();
        }
        Map<String, Map<String, Double>> chartMap = (Map<String, Map<String, Double>>) map.getOrDefault("chart", new Object());
        if (MapUtils.isNotEmpty(chartMap)) {
            // 媒体分析返回的图表，需要翻译媒体ID/广告位ID
            if (IntegrationConstant.Analysis.ANALYSIS_TYPE_MEDIA.equals(analysisType)) {
                map.put("chart", ticketAnalysisHelper.commonAnalysisChat(chartMap, analysisType, paramsDTO.getMediaGroupType()));
            } else if (IntegrationConstant.Analysis.ANALYSIS_TYPE_TAG_TYPE.equals(analysisType)) {
                map.put("chart", ticketAnalysisHelper.commonAnalysisChat(chartMap, analysisType, ""));
            }
        }
        // 翻译 page list中需要被翻译得字段
        try {
            ObjectMapper objectMapper = new ObjectMapper();
            switch (analysisType) {
                case IntegrationConstant.Analysis.ANALYSIS_TYPE_TREND:
                    Page<com.bxm.datapark.facade.ticket.model.vo.analysis.AnalysisTrendVO> trendVOPage = objectMapper.readValue(JSON.toJSONString(map.get("page")), new TypeReference<Page<com.bxm.datapark.facade.ticket.model.vo.analysis.AnalysisTrendVO>>(){});
                    map.put("page", ticketAnalysisHelper.convertAndBuildByTrend(trendVOPage));
                    break;
                case IntegrationConstant.Analysis.ANALYSIS_TYPE_MEDIA:
                    Page<com.bxm.datapark.facade.ticket.model.vo.analysis.AnalysisMediaVO> mediaVOPage = objectMapper.readValue(JSON.toJSONString(map.get("page")), new TypeReference<Page<com.bxm.datapark.facade.ticket.model.vo.analysis.AnalysisMediaVO>>(){});
                    map.put("page", ticketAnalysisHelper.convertAndBuildByMedia(mediaVOPage));
                    break;
                case IntegrationConstant.Analysis.ANALYSIS_TYPE_TAG_TYPE:
                    Page<com.bxm.datapark.facade.ticket.model.vo.analysis.AnalysisTagTypeVO> tagTypeVOPage = objectMapper.readValue(JSON.toJSONString(map.get("page")), new TypeReference<Page<com.bxm.datapark.facade.ticket.model.vo.analysis.AnalysisTagTypeVO>>(){});
                    map.put("page", ticketAnalysisHelper.convertAndBuildByTagType(tagTypeVOPage));
                    break;
                case IntegrationConstant.Analysis.ANALYSIS_TYPE_REGION:
                    Page<com.bxm.datapark.facade.ticket.model.vo.analysis.AnalysisRegionVO> regionVOPage = objectMapper.readValue(JSON.toJSONString(map.get("page")), new TypeReference<Page<com.bxm.datapark.facade.ticket.model.vo.analysis.AnalysisRegionVO>>(){});
                    map.put("page", ticketAnalysisHelper.convertAndBuildByRegion(regionVOPage));
                    break;
                case IntegrationConstant.Analysis.ANALYSIS_TYPE_APP_OS:
                    Page<com.bxm.datapark.facade.ticket.model.vo.analysis.AnalysisAppOSVO> appOSVOPage = objectMapper.readValue(JSON.toJSONString(map.get("page")), new TypeReference<Page<com.bxm.datapark.facade.ticket.model.vo.analysis.AnalysisAppOSVO>>(){});
                    map.put("page", ticketAnalysisHelper.convertAndBuildByAppOS(appOSVOPage));
                    break;
                case IntegrationConstant.Analysis.ANALYSIS_TYPE_PAYMENT_TYPE:
                    Page<com.bxm.datapark.facade.ticket.model.vo.analysis.AnalysisPaymentTypeVO> paymentTypeVOPage = objectMapper.readValue(JSON.toJSONString(map.get("page")), new TypeReference<Page<com.bxm.datapark.facade.ticket.model.vo.analysis.AnalysisPaymentTypeVO>>(){});
                    map.put("page", ticketAnalysisHelper.convertAndBuildByPaymentType(paymentTypeVOPage));
                    break;
                default:
                    return null;
            }
        } catch (IOException e) {
            log.error("【报表系统--数据统计，处理datapark返回的数据异常！】", e);
            return Collections.emptyMap();
        }
        return map;
    }


    /**
     * Description: DTO转换
     * JDK version used: <JDK1.8>
     * Author： hxpeng
     * Create Date： 2018/8/17 18:42
     */
    private com.bxm.datapark.facade.ticket.model.dto.AnalysisParamsDTO convert(AnalysisParamsDTO paramsDTO) {
        com.bxm.datapark.facade.ticket.model.dto.AnalysisParamsDTO analysisParamsDTO = new com.bxm.datapark.facade.ticket.model.dto.AnalysisParamsDTO();
        analysisParamsDTO.setSortType(paramsDTO.getSortType());
        analysisParamsDTO.setSortFieldName(paramsDTO.getSortFieldName());
        analysisParamsDTO.setRegionGroupType(paramsDTO.getRegionGroupType());
        analysisParamsDTO.setMediaGroupType(paramsDTO.getMediaGroupType());
        if (null != paramsDTO.getTicketId()) {
            analysisParamsDTO.setTicketId(String.valueOf(paramsDTO.getTicketId()));
        }
        analysisParamsDTO.setMediaId(paramsDTO.getMediaId());
        analysisParamsDTO.setAdPositionId(paramsDTO.getAdPositionId());
        analysisParamsDTO.setPageNum(paramsDTO.getPageNum());
        analysisParamsDTO.setPageSize(paramsDTO.getPageSize());
        analysisParamsDTO.setStartTime(paramsDTO.getStartTime());
        analysisParamsDTO.setEndTime(paramsDTO.getEndTime());
        return analysisParamsDTO;
    }

}
