package com.bxm.adscounter.api.ticket;

import com.alibaba.fastjson.JSONObject;
import com.bxm.adscounter.api.ticket.conversion.Conversion;
import com.bxm.adscounter.api.ticket.conversion.ConversionService;
import com.bxm.adscounter.facade.model.AdGroupCountMsg;
import com.bxm.adscounter.model.constant.AdGroupCounterConstant;
import com.bxm.warcar.utils.HttpUtils;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.response.ResponseModel;
import com.bxm.warcar.utils.response.ResponseModelFactory;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpHeaders;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import java.io.BufferedReader;
import java.io.IOException;
import java.util.Map;
import java.util.Objects;

/**
 * <p> 广告组统计接口 </p>
 *
 * @Author: JandMin
 * @since: 1.0.0
 * @Date: 2018/8/9
 * @author allen
 * @date 2022-04-21
 */
@Slf4j
@RestController
public class AdGroupController {

    private final static Logger LOGGER = LoggerFactory.getLogger(AdGroupController.class);

    private final ConversionService conversionService;

    public AdGroupController(ConversionService conversionService) {
        this.conversionService = conversionService;
    }

    @RequestMapping(value = "/shop/countInfo", method = RequestMethod.POST)
    public ResponseModel<String> adGroupCountInfo(AdGroupCountMsg adGroupCountMsg, HttpServletRequest request) {
        ResponseModel<String> result;
        try {
            adGroupCountMsg = fill(adGroupCountMsg, request);

            log.info("Trace: {}", JsonHelper.convert(adGroupCountMsg));

            if (StringUtils.isNotBlank(adGroupCountMsg.getModeltype()) && "7".equals(adGroupCountMsg.getModeltype())){
                sendOpenLog(adGroupCountMsg, request);
            }

            result = ResponseModelFactory.SUCCESS("success");
        } catch (Exception e) {
            result = ResponseModelFactory.FAILED500();
            LOGGER.error("shop/countInfo error request =" + adGroupCountMsg.toString(), e);
        }
        return result;
    }

    @RequestMapping(value = "/shop/effectEvent")
    public ResponseModel<String> adGroupAdsCountInfo(AdGroupCountMsg adGroupCountMsg, HttpServletRequest request) {
        ResponseModel<String> result;
        try {
            adGroupCountMsg = fill(adGroupCountMsg, request);
            sendOpenLog(adGroupCountMsg, request);
            result = ResponseModelFactory.SUCCESS("success");
        } catch (Exception e) {
            result = ResponseModelFactory.FAILED500();
            LOGGER.error("shop/effectEvent error request =" + adGroupCountMsg.toString(), e);
        }
        return result;
    }

    @RequestMapping(value = "/shop/customizeEffectEvent")
    public ResponseEntity<Object> customizeResponse(AdGroupCountMsg adGroupCountMsg, HttpServletRequest request) {
        Map<String, Object> res = Maps.newHashMap();
        try {
            adGroupCountMsg = fill(adGroupCountMsg, request);
            this.sendOpenLog(adGroupCountMsg, request);
            res.put("code", 0);
        } catch (Exception e) {
            LOGGER.error("shop/effectEvent error request =" + adGroupCountMsg.toString(), e);
            res.put("code", 1);
        }
        return ResponseEntity.ok(res);
    }

    private void sendOpenLog(AdGroupCountMsg adGroupCountMsg, HttpServletRequest request) {
        Conversion conversion = createConversion(adGroupCountMsg, request);
        conversionService.sendOpenLog(conversion);
    }

    private Conversion createConversion(AdGroupCountMsg adGroupCountMsg, HttpServletRequest request) {
        return Conversion.builder()
                        .bxmId(adGroupCountMsg.getBxm_id())
                        .ip(adGroupCountMsg.getIpAddress())
                        .ua(request.getHeader(HttpHeaders.USER_AGENT))
                        .refer(adGroupCountMsg.getReferrer())
                        .status(adGroupCountMsg.getStatus())
                        .type(Objects.toString(adGroupCountMsg.getType(), "0"))
                        .transformType(getTransformType(adGroupCountMsg))
                        .build();
    }

    private String getTransformType(AdGroupCountMsg adGroupMsg) {
        String transformType = adGroupMsg.getTransformType();
        if (StringUtils.isBlank(transformType)) {
            transformType = adGroupMsg.getEvent_type();
        }
        return transformType;
    }

    private static AdGroupCountMsg fill(AdGroupCountMsg adGroupCountMsg, HttpServletRequest request) {
        if (null == adGroupCountMsg) {
            adGroupCountMsg = new AdGroupCountMsg();
        }
        //获取IP
        adGroupCountMsg.setIpAddress(HttpUtils.getIpFromHeader(request));

        //从请求头中获取参数
        StringBuilder sb = new StringBuilder();
        try (BufferedReader reader = request.getReader();) {
            char[] buff = new char[1024];
            int len;
            while ((len = reader.read(buff)) != -1) {
                sb.append(buff, 0, len);
            }
        } catch (IOException e) {
            LOGGER.error("[HttpParamUtils] checkAdGroupCountMsg error ======", e);
        }
        String params = sb.toString();

        //把为空的参数再赋值
        if (StringUtils.isNotBlank(params)) {
            JSONObject json = JSONObject.parseObject(params);
            if (null == adGroupCountMsg.getParam()) {
                adGroupCountMsg.setParam(json.getString("param"));
            }
            if (null == adGroupCountMsg.getPhone()) {
                adGroupCountMsg.setPhone(json.getString("phone"));
            }
            if (null == adGroupCountMsg.getIdcardnum()) {
                adGroupCountMsg.setIdcardnum(json.getString("idcardnum"));
            }
            if (null == adGroupCountMsg.getShop_code()) {
                adGroupCountMsg.setShop_code(json.getString("shop_code"));
            }
            if (null == adGroupCountMsg.getBxm_id()) {
                String countId = json.getString("bxm_id");
                adGroupCountMsg.setBxm_id(StringUtils.isBlank(countId) || "null".equals(countId) ? "0" : countId);
            }
            if (null == adGroupCountMsg.getStatus()) {
                String status = json.getString("status");
                adGroupCountMsg.setStatus(StringUtils.isBlank(status) || "null".equals(status) ? AdGroupCounterConstant.DEFAULT_STATUS : status);
            }
            if (null == adGroupCountMsg.getModeltype()) {
                String modelType = json.getString("modeltype");
                adGroupCountMsg.setModeltype(StringUtils.isBlank(modelType) || "null".equals(modelType) ?
                        AdGroupCounterConstant.DEFAULT_MODEL_TYPE : modelType);
            }
            if (null == adGroupCountMsg.getSh() ){
                adGroupCountMsg.setSh(json.getInteger("sh"));
            }
            if (null == adGroupCountMsg.getSw() ){
                adGroupCountMsg.setSw(json.getInteger("sw"));
            }
            if (null == adGroupCountMsg.getTs() ){
                adGroupCountMsg.setTs(json.getLong("ts"));
            }
            if (null == adGroupCountMsg.getLocaurl() ){
                adGroupCountMsg.setLocaurl(json.getString("locaurl"));
            }
        }

        //类型参数处理，防止异常
        if (!StringUtils.isNumeric(adGroupCountMsg.getModeltype())) {
            adGroupCountMsg.setModeltype(AdGroupCounterConstant.DEFAULT_MODEL_TYPE);
        }

        //状态参数处理，防止异常
        if (!StringUtils.isNumeric(adGroupCountMsg.getStatus())) {
            adGroupCountMsg.setStatus(AdGroupCounterConstant.DEFAULT_STATUS);
        }
        adGroupCountMsg.setReferrer(request.getHeader(HttpHeaders.REFERER));
        return adGroupCountMsg;
    }
}
