package com.bxm.adscounter.rtb.common.aop;

import com.bxm.adscounter.rtb.common.ClickTracker;
import com.bxm.adscounter.rtb.common.RtbIntegration;
import com.bxm.adscounter.rtb.common.control.RatioControlRtbIntegration;
import com.bxm.adscounter.rtb.common.control.ticket.TicketRatioControl;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.warcar.utils.JsonHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ArrayUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.core.Ordered;
import redis.clients.jedis.JedisPool;

import java.util.Objects;

/**
 * @author allen
 * @date 2022-07-12
 * @since 1.0
 */
@Slf4j
@Aspect
public class ControlRtbIntegrationAspect implements Ordered {

    private final RatioControlRtbIntegration ratioControlRtbIntegration;
    private final JedisPool jedisPool;
    private final TicketRatioControl ticketRatioControl;

    public ControlRtbIntegrationAspect(RatioControlRtbIntegration ratioControlRtbIntegration, JedisPool jedisPool, TicketRatioControl ticketRatioControl) {
        this.ratioControlRtbIntegration = ratioControlRtbIntegration;
        this.jedisPool = jedisPool;
        this.ticketRatioControl = ticketRatioControl;
    }

    @Override
    public int getOrder() {
        return HIGHEST_PRECEDENCE;
    }

    @Pointcut("this(com.bxm.adscounter.rtb.common.RtbIntegration) && execution(com.bxm.adscounter.rtb.common.feedback.FeedbackResponse doFeedback(com.bxm.adscounter.rtb.common.feedback.FeedbackRequest))")
    public void pointcut() {}

    @Around("pointcut()")
    public Object around(ProceedingJoinPoint point) throws Throwable {
        if (log.isDebugEnabled()) {
            log.debug("execute: {}", getClass());
        }
        Object target = point.getThis();
        if (!(target instanceof RtbIntegration)) {
            return point.proceed();
        }
        Object[] args = point.getArgs();
        if (ArrayUtils.isEmpty(args)) {
            return point.proceed();
        }
        Object firstArg = args[0];
        if (!(firstArg instanceof FeedbackRequest)) {
            return point.proceed();
        }
        FeedbackRequest request = (FeedbackRequest) firstArg;
        RtbIntegration rtbIntegration = (RtbIntegration) target;

        PositionRtb config = request.getConfig();
        if (Objects.isNull(config)) {
            // oCPX 通用上报与回传，不会经过我们的系统，但使用了 RtbIntegration。
            // 在 com.bxm.adscounter.service.openlog.common.feedback.impl
            // 在 https://bxmrds.yuque.com/mizhsy/qc669f/rgie7h
            return point.proceed();
        }

        // 券扣量控制
        try {
            boolean onTicketRatioControl = ticketRatioControl.onControl(request, rtbIntegration);
            if (onTicketRatioControl) {
                return FeedbackResponse.ofSuccess();
            }
        } catch (Exception e) {
            log.error("Ticket Ratio control invoke occur exception! " + JsonHelper.convert(request), e);
        }

        // 是否被控制
        boolean controlled = false;
        try {
            if (config.isCvrOrDeductionControl()) {
                controlled = ratioControlRtbIntegration.onFeedbackControl(rtbIntegration, request);
            }
        } catch (Exception e) {
            // 如果控制发生异常，不能影响正常回传，还是要继续回传。
            log.error("Feedback control invoke occur exception! " + JsonHelper.convert(request), e);
        }
        if (!controlled) {
            return point.proceed();
        }
        return FeedbackResponse.ofSuccess();
    }
    public JedisPool getJedisPool() {
        return jedisPool;
    }

}
