package com.bxm.adscounter.rtb.common.aop;

import com.bxm.adscounter.rtb.common.RtbIntegration;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.ArrayUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.core.Ordered;
import org.springframework.util.CollectionUtils;

import java.util.List;

/**
 * @author huxiao
 * @date 2022-05-13
 */
@Slf4j
@Aspect
public class RtbIntegrationAspect implements Ordered {

    private final List<RtbFeedbackInterceptor> interceptors;

    public RtbIntegrationAspect(List<RtbFeedbackInterceptor> interceptors) {
        this.interceptors = interceptors;
    }

    @Override
    public int getOrder() {
        return LOWEST_PRECEDENCE;
    }

    @Pointcut("this(com.bxm.adscounter.rtb.common.RtbIntegration) && execution(com.bxm.adscounter.rtb.common.feedback.FeedbackResponse doFeedback(..))")
    public void pointcut() {}

    @Around("pointcut()")
    public Object around(ProceedingJoinPoint point) throws Throwable {
        if (log.isDebugEnabled()) {
            log.debug("execute: {}", getClass());
        }
        Object target = point.getTarget();
        if (!(target instanceof RtbIntegration)) {
            return point.proceed();
        }

        if (CollectionUtils.isEmpty(interceptors)) {
            return point.proceed();
        }

        RtbIntegration integration = (RtbIntegration) target;
        FeedbackRequest request = null;
        FeedbackResponse response = null;
        try {
            Object[] args = point.getArgs();
            if (ArrayUtils.isEmpty(args)) {
                return point.proceed();
            }
            Object arg = args[0];
            if (!(arg instanceof FeedbackRequest)) {
                return point.proceed();
            }
            request = (FeedbackRequest) arg;

            for (RtbFeedbackInterceptor interceptor : interceptors) {
                try {
                    interceptor.doRequest(integration, request);
                } catch (Exception e) {
                    log.error("doRequest: ", e);
                }
            }

            Object proceed = point.proceed();

            if (!(proceed instanceof FeedbackResponse)) {
                return proceed;
            }

            response = (FeedbackResponse) proceed;

            if (response.isSuccess()) {
                for (RtbFeedbackInterceptor interceptor : interceptors) {
                    try {
                        interceptor.doSuccess(integration, request, response);
                    } catch (Exception e) {
                        log.error("doSuccess: ", e);
                    }
                }
            } else {
                for (RtbFeedbackInterceptor interceptor : interceptors) {
                    try {
                        interceptor.doFail(integration, request, response);
                    } catch (Exception e) {
                        log.error("doFail: ", e);
                    }
                }
            }

            return proceed;
        } catch (Throwable throwable) {
            for (RtbFeedbackInterceptor interceptor : interceptors) {
                try {
                    interceptor.doException(integration, request, response, throwable);
                } catch (Exception e) {
                    log.error("doException: ", e);
                }
            }
            throw throwable;
        }
    }
}
