package com.bxm.adscounter.rtb.common.aop.interceptor;

import com.bxm.adscounter.rtb.common.RtbIntegration;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.RtbIntegrationFactory;
import com.bxm.adscounter.rtb.common.aop.RtbFeedbackInterceptor;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.google.common.collect.Lists;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.binder.MeterBinder;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.DisposableBean;

import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 重试的拦截器。
 *
 * @author allen
 * @date 2022-12-19
 * @since 1.0
 */
@Slf4j
public class RetryInterceptor implements RtbFeedbackInterceptor, DisposableBean, MeterBinder {

    private final RtbIntegrationFactory factory;
    private final BlockingQueue<Retry> queue;
    private final ScheduledThreadPoolExecutor timer;
    private final int maxRetryTimes;

    public RetryInterceptor(RtbIntegrationFactory factory, int delay, int capacity, int maxRetryTimes) {
        this.factory = factory;
        this.queue = new LinkedBlockingQueue<>(capacity);
        this.maxRetryTimes = maxRetryTimes;
        this.timer = new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("retry-scheduler"));
        this.timer.scheduleWithFixedDelay(() -> {
            try {
                List<Retry> retries = Lists.newArrayList();
                int i = queue.drainTo(retries);
                if (i > 0) {
                    if (log.isDebugEnabled()) {
                        log.debug("Wait for try size: {}", i);
                    }
                    for (Retry retry : retries) {
                        doRetry(retry);
                    }
                }
            } catch (Exception e) {
                log.error("scheduled: ", e);
            }
        }, delay, delay, TimeUnit.SECONDS);
    }

    @Override
    public void bindTo(MeterRegistry registry) {
        registry.gauge("rtb.retry.queue.size", 0, value -> queue.size());
    }

    @Override
    public void destroy() {
        this.timer.shutdownNow();
    }

    private void doRetry(Retry retry) {
        RtbIntegration integration = retry.getIntegration();
        FeedbackRequest request = retry.getRequest();
        int times = request.getRetryTimes();
        if (times >= maxRetryTimes) {
            return;
        }
        try {
            // 重试次数 +1
            request.incrementRetryTimes();
            // 执行回传
            RtbIntegration proxyInstance = getRtbIntegrationCglibProxyInstance(integration);
            proxyInstance.doFeedback(request, 9);
        } catch (RtbIntegrationException ignored) {
        }
    }

    private RtbIntegration getRtbIntegrationCglibProxyInstance(RtbIntegration integration) {
        return factory.get(integration.rtb());
    }

    @Override
    public void doRequest(RtbIntegration integration, FeedbackRequest request) {
    }

    @Override
    public void doSuccess(RtbIntegration integration, FeedbackRequest request, FeedbackResponse response) {
    }

    @Override
    public void doFail(RtbIntegration integration, FeedbackRequest request, FeedbackResponse response) {
    }

    @Override
    public void doException(RtbIntegration integration, FeedbackRequest request, FeedbackResponse response, Throwable throwable) {
        if (!(throwable instanceof RtbIntegrationException)) {
            return;
        }
        RtbIntegrationException exception = (RtbIntegrationException) throwable;
        switch (exception.getFailType()) {
            case IoException:
            case ResolveProtocolException:
                if (!queue.offer(new Retry(integration, request))) {
                    log.error("Oops!! The queue size full!");
                }
                break;
            default:
                break;
        }
    }

    @Data
    @AllArgsConstructor
    private static class Retry {
        private RtbIntegration integration;
        private FeedbackRequest request;
    }
}
