package com.bxm.adscounter.rtb.common.control;

import com.bxm.adscounter.rtb.common.RtbIntegration;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.control.deduction.ConversionLevel;
import com.bxm.adscounter.rtb.common.control.deduction.DeductionControl;
import com.bxm.adscounter.rtb.common.control.deduction.DeductionControlConfig;
import com.bxm.adscounter.rtb.common.control.deduction.RedisDeductionControlImpl;
import com.bxm.adscounter.rtb.common.event.RtbDeductionEvent;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.service.PositionRtbService;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.openlog.sdk.consts.Inads;
import com.bxm.warcar.integration.eventbus.AsyncEventPark;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.bxm.warcar.utils.SafeMapHelper;
import com.bxm.warcar.utils.TypeHelper;
import io.micrometer.core.instrument.MeterRegistry;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import redis.clients.jedis.JedisPool;

import java.time.LocalTime;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Predicate;

import static com.bxm.adscounter.rtb.common.control.LocalDateTimeUtils.parseTime;

/**
 * @author tangx
 * @date 2022/8/4 16:00
 * @see DefaultRatioControlRtbIntegrationImpl
 */
@Slf4j
@Deprecated
public class DefaultDeductionControlRtbIntegrationImpl implements DeductionControlRtbIntegration{

    private final ScheduledThreadPoolExecutor refreshExecutor = new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("refresh"));

    private final ConcurrentHashMap<String /* 广告位 + adGroupId + convLevel */, String /* 控制器 Key (Dimension) */ > referenced = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Long /* 配置ID */, Set<String> /* 控制器 Key (Dimension) */> mapping = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String /* 控制器 Key (Dimension) */, DeductionControl> controlExecutors = new ConcurrentHashMap<>();

    private final JedisPool jedisPool;
    private final PositionRtbService service;
    private MeterRegistry registry;
    private final AsyncEventPark asyncEventPark;

    public DefaultDeductionControlRtbIntegrationImpl(JedisPool jedisPool, PositionRtbService service,
                                                     AsyncEventPark asyncEventPark) {
        this.jedisPool = jedisPool;
        this.service = service;
        this.asyncEventPark = asyncEventPark;
        this.refreshExecutor.scheduleWithFixedDelay(() -> {
            try {
                this.refreshControlForMapping();
            } catch (Exception e) {
                log.error("occur ex: ", e);
            }
        }, 1, 1, TimeUnit.MINUTES);
    }

    @Override
    public boolean onFeedbackControl(RtbIntegration instance, FeedbackRequest request) {
        String adGroupId = request.getAdGroupId();
        if (StringUtils.isBlank(adGroupId)) {
            return false;
        }
        String tagId = request.getKeyValueMap().getFirst(Inads.Param.TAGID);
        PositionRtb config = request.getConfig();
        if (Objects.isNull(config)) {
            return false;
        }
        PositionRtb.CvrControl hit = chooseCvrControl(config, adGroupId);
        if (Objects.isNull(hit)) {
            return false;
        }
        ConversionLevel convLevel = request.isDeepConversion() ? ConversionLevel.DEEP : ConversionLevel.SHALLOW;
        // 控制器
        DeductionControl control = createIfNecessary(instance, tagId, adGroupId, hit, convLevel);
        // 处理是否回传
        control.push(JsonHelper.convert(request));
        return true;
    }

    private PositionRtb.CvrControl chooseCvrControl(PositionRtb config, String adGroupId) {
        if (!config.isDeductionControl()) {
            return null;
        }
        List<PositionRtb.CvrControl> rateControls = config.getRateControls();
        if (CollectionUtils.isEmpty(rateControls)) {
            return null;
        }
        PositionRtb.CvrControl hit = null;
        for (PositionRtb.CvrControl cvrControl : rateControls) {
            if (!cvrControl.isEnable()) {
                continue;
            }
            String theAdGroupId = cvrControl.getAdGroupId();
            // start >= now <= end
            String startTime = cvrControl.getStartTime();
            String endTime = cvrControl.getEndTime();
            if (!isValidTime(startTime, endTime)) {
                continue;
            }
            if (StringUtils.isBlank(theAdGroupId)) {
                // 全局定向
                hit = cvrControl;
            }
            if (StringUtils.equals(adGroupId, theAdGroupId)) {
                // [优先] 广告组定向
                hit = cvrControl;
                break;
            }
        }
        return hit;
    }

    private boolean isValidTime(String startTime, String endTime) {
        LocalTime start = parseTime(startTime);
        LocalTime end = parseTime(endTime);
        if (Objects.isNull(start) || Objects.isNull(end)) {
            return false;
        }
        LocalTime now = LocalTime.now().withNano(0);
        // start >= now <= end
        return (now.equals(start) || now.equals(end) || (now.isAfter(start) && now.isBefore(end)));
    }

    private DeductionControl createIfNecessary(RtbIntegration instance, String tagId, String adGroupId, PositionRtb.CvrControl hit, ConversionLevel level) {
        String key = ControlUtils.createKey(tagId, adGroupId, TypeHelper.castToString(hit.getId()), hit.getStartTime(), hit.getEndTime(), level.name());
        return SafeMapHelper.get(controlExecutors, key, new SafeMapHelper.InitializingValue<DeductionControl>() {
            @Override
            public DeductionControl initializing() {
                // 有命中的控制条目
                double ratio = level == ConversionLevel.DEEP ? hit.getDeepCvr() : hit.getCvr();
                DeductionControlConfig deductionControlConfig = new DeductionControlConfig()
                        .setDimension(key)
                        .setRatio(ratio)
                        .setHitConfigId(hit.getId())
                        .setLevel(level)
                        .setConsumer(new Consumer<String>() {
                            @Override
                            public void accept(String s) {
                                FeedbackRequest feedbackRequest = JsonHelper.convert(s, FeedbackRequest.class);
                                doFeedbackQuietly(instance, feedbackRequest);
                            }
                        })
                        .setDeductionConsumer(new Consumer<String>() {
                            @Override
                            public void accept(String s) {
                                FeedbackRequest feedbackRequest = JsonHelper.convert(s, FeedbackRequest.class);
                                asyncEventPark.post(new RtbDeductionEvent(s, instance.rtb(), feedbackRequest, null));
                            }
                        })
                        ;
                DeductionControl deductionControl = new RedisDeductionControlImpl(jedisPool, deductionControlConfig, registry);
                // 保存当前配置对应的控制器 Key
                SafeMapHelper.get(mapping, hit.getId(), (SafeMapHelper.InitializingValue<Set<String>>) HashSet::new).add(key);
                // 保存引用
                String previous = referenced.put(tagId + adGroupId + level.name(), key);
                if (Objects.nonNull(previous)) {
                    log.info("Deduction control {} reference changed to {}", previous, key);
                }
                return deductionControl;
            }
        });
    }

    private void refreshControlForMapping() {
        if (log.isDebugEnabled()) {
            log.debug("Starting refresh deduction Controller...");
        }
        Set<String> alive = new HashSet<>(referenced.values());
        mapping.forEach(new BiConsumer<Long, Set<String>>() {
            @Override
            public void accept(Long id, Set<String> keys) {
                keys.forEach(key -> {
                    if (log.isDebugEnabled()) {
                        log.debug("[{}] Checking {}", id, key);
                    }
                    String tagId = ControlUtils.splitKey(key)[0];
                    PositionRtb config = service.get(tagId);
                    if (Objects.isNull(config)) {
                        // 配置无效时，关闭
                        close(id, true, CloseReason.Illegal_Config);
                        return;
                    }

                    List<PositionRtb.CvrControl> controls = config.getRateControls();
                    if (CollectionUtils.isEmpty(controls)) {
                        // 控制配置无效时，关闭
                        close(id, true, CloseReason.No_Control);
                        return;
                    }

                    if (!config.isDeductionControl()) {
                        // 不是扣量回传，关闭
                        close(id, false, CloseReason.Not_Deduction);
                        return;
                    }

                    if (!alive.contains(key)) {
                        // 已经没有引用，关闭
                        close(id, true, CloseReason.Dead);
                        return;
                    }

                    for (PositionRtb.CvrControl cfg : controls) {
                        if (!Objects.equals(id, cfg.getId())) {
                            continue;
                        }
                        if (!cfg.isEnable()) {
                            // 状态关闭时，关闭
                            close(id, false, CloseReason.Disabled);
                            continue;
                        }
                        String startTime = cfg.getStartTime();
                        String endTime = cfg.getEndTime();
                        if (!isValidTime(startTime, endTime)) {
                            // 无效时间，关闭
                            close(id, true, CloseReason.Invalid_Time);
                            continue;
                        }

                        // 刷新配置
                        refresh(id, cfg);
                    }
                });
            }
        });
    }

    private void refresh(Long id, PositionRtb.CvrControl cfg) {
        Set<String> executors = mapping.get(id);
        if (CollectionUtils.isNotEmpty(executors)) {
            executors.forEach(key -> {
                DeductionControl control = controlExecutors.get(key);
                if (Objects.nonNull(control)) {
                    DeductionControlConfig config = control.getConfig();
                    if (config.getLevel() == ConversionLevel.DEEP) {
                        if (cfg.getDeepCvr() != config.getRatio()) {
                            if (log.isInfoEnabled()) {
                                log.info("[{} - {}] deep rate value {} changed to {}", id, key, config.getRatio(), cfg.getDeepCvr());
                            }
                            config.setRatio(cfg.getDeepCvr());
                        }
                    } else {
                        if (cfg.getCvr() != config.getRatio()) {
                            if (log.isInfoEnabled()) {
                                log.info("[{} - {}] shallow rate value {} changed to {}", id, key, config.getRatio(), cfg.getCvr());
                            }
                            config.setRatio(cfg.getCvr());
                        }
                    }
                }
            });
        }
    }

    private void close(Long id, Boolean isDeleteRedisData, CloseReason reason) {
        Set<String> executors = mapping.get(id);
        if (CollectionUtils.isNotEmpty(executors)) {
            executors.removeIf(new Predicate<String>() {
                @Override
                public boolean test(String key) {
                    DeductionControl control = controlExecutors.get(key);
                    if (Objects.nonNull(control)) {
                        if (isDeleteRedisData) {
                            control.delete();
                        }
                    }
                    controlExecutors.remove(key);
                    if (log.isInfoEnabled()) {
                        log.info("[{}] Close control: {} - {}", reason, key, control);
                    }
                    return true;
                }
            });
        }
    }

    @Override
    public void bindTo(MeterRegistry registry) {
        this.registry = registry;
    }

    private void doFeedbackQuietly(RtbIntegration instance, FeedbackRequest request) {
        try {
            instance.doFeedback(request, 1);
        } catch (RtbIntegrationException e) {
            log.error("doFeedbackQuietly: ", e);
        }
    }

    enum CloseReason {

        /**
         * Illegal position rtb config
         */
        Illegal_Config,

        No_Control,

        Not_Deduction,

        Dead,

        Disabled,

        Invalid_Time
    }
}
