package com.bxm.adscounter.rtb.common.control.cpa;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.adscounter.rtb.common.DataFetchFailException;
import com.bxm.adscounter.rtb.common.data.AdGroupData;
import com.bxm.adscounter.rtb.common.data.DorisAdGroupHourlyData;
import com.bxm.adscounter.rtb.common.data.Parameter;
import com.bxm.adscounter.rtb.common.impl.kuaishou.KuaishouDataFetcher;
import com.bxm.adscounter.rtb.common.mapper.DorisMapper;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.adsprod.facade.ticket.rtb.PreSummaryFeedbackConfig;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.utils.TypeHelper;
import lombok.Data;
import lombok.experimental.Accessors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.logging.log4j.util.Strings;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * 累加前置判断
 * @author tangxiao
 * @date 2023/10/16
 * @since 1.0
 */
@Slf4j
public class PreSummaryFeedbackHelper {
    private static final int ONE_DAY_SEC = TypeHelper.castToInt(Duration.ofDays(1).getSeconds());

    private final Counter counter;
    private final Fetcher fetcher;
    private final KuaishouDataFetcher kuaishouDataFetcher;
    private final DorisMapper dorisMapper;

    public PreSummaryFeedbackHelper(Counter counter, Fetcher fetcher, KuaishouDataFetcher kuaishouDataFetcher, DorisMapper dorisMapper) {
        this.counter = counter;
        this.fetcher = fetcher;
        this.kuaishouDataFetcher = kuaishouDataFetcher;
        this.dorisMapper = dorisMapper;
    }

    /**
     * 是否满足累加前置回传
     * @param positionRtb
     * @param adGroupId
     * @return
     */
    public boolean needFeedback(PositionRtb positionRtb, String adGroupId) {
        if (!positionRtb.isPreSummaryFeedback()) {
            return false;
        }
        String positionId = positionRtb.getPositionId();

        FinalActiveParameter finalActiveParameter = getFinalActiveParameter(positionId, adGroupId, positionRtb);

        // 判断使用账户维度的计数
        String dimension = finalActiveParameter.isUseAccountDimension() ? PreSummaryFeedbackConfig.ACCOUNT_FIELD : adGroupId;
        BigDecimal lossThreshold = finalActiveParameter.getLossThreshold();

        // 首先判断是否超过亏损阈值，未超过：直接回传，超过：继续后续判断
        if(needFeedbackIfAchieveLossThreshold(lossThreshold, positionRtb, positionId, dimension)){
            log.info("PreSummary {} - {} needFeedbackIfAchieveLossThreshold return true. finalActiveParameter = {}", positionId, dimension, JSON.toJSONString(finalActiveParameter));
            return true;
        }

        // 代表这个广告组没有配置
        Integer feedbackMaxNum = finalActiveParameter.getFeedbackMaxNum();
        if (feedbackMaxNum == null) {
            return false;
        }

        // 先获取判断
        Integer count = Optional.ofNullable(fetcher.hfetch(hashPreSummaryFeedbackCount(positionId), dimension, Integer.class)).orElse(0);
        if (count >= feedbackMaxNum) {
            return false;
        }

        // 次数加一再次判断
        Long afterIncr = counter.hincrementAndGet(hashPreSummaryFeedbackCount(positionId), dimension, ONE_DAY_SEC);
        return afterIncr <= feedbackMaxNum;
    }

    private boolean needFeedbackIfAchieveLossThreshold(BigDecimal lossThreshold, PositionRtb positionRtb, String positionId, String adGroupId) {
        if (lossThreshold == null) {
            return false;
        }
        // 目前只有快手支持亏损阈值
        if (!Objects.equals(PositionRtb.AllSourceType.KUAISHOU, positionRtb.getSourceType())) {
            return false;
        }
        BigDecimal count = Optional.ofNullable(fetcher.hfetch(hashPreSummaryFeedbackLossThresholdCount(positionId), adGroupId, BigDecimal.class)).orElse(BigDecimal.ZERO);
        log.info("[{} - {}] maxLossThreshold: {}, actualLossThreshold:{}", positionId, adGroupId, lossThreshold, count);
        if (lossThreshold.compareTo(count) < 0) {
            return false;
        }

        BigDecimal income = getAdGroupIncome(positionId, adGroupId);
        if (income == null) {
            return false;
        }
        BigDecimal charge = getAdGroupCharge(positionRtb, adGroupId);
        if (charge == null) {
            return false;
        }

        BigDecimal profit = computeProfit(income, charge);
        Double afterIncr = counter.hincrFloatByAndGet(hashPreSummaryFeedbackLossThresholdCount(positionId), adGroupId, profit.doubleValue());
        return lossThreshold.compareTo(BigDecimal.valueOf(afterIncr)) > 0;
    }

    private BigDecimal getAdGroupCharge(PositionRtb positionRtb, String adGroupId) {
        if (isAccountDimension(adGroupId)) {
            adGroupId = null;
        }
        Parameter parameter = Parameter.builder()
                .advertiserId(positionRtb.getCustomerId())
                .tagId(positionRtb.getPositionId())
                .adGroupId(adGroupId)
                .build();

        List<AdGroupData> adGroupData = null;
        try {
            adGroupData = kuaishouDataFetcher.fetchCurrentHourData(parameter);
        } catch (DataFetchFailException e) {
            log.error("preSummary getAdGroupCharge occur error", e);
            return null;
        }

        BigDecimal charge = null;
        if (CollectionUtils.isNotEmpty(adGroupData)) {
            charge = adGroupData.stream()
                    .map(AdGroupData::getCharge)
                    .filter(Objects::nonNull)
                    .reduce(BigDecimal.ZERO, BigDecimal::add);
        }
        return charge;
    }

    private BigDecimal getAdGroupIncome(String positionId, String adGroupId) {
        BigDecimal income = null;
        if (isAccountDimension(adGroupId)) {
            adGroupId = null;
        }
        List<DorisAdGroupHourlyData> dorisAdGroupHourlyData = dorisMapper.getAdGroupHourlyData(positionId, adGroupId, DateHelper.format(DateHelper.PATTERN_STR10), DateHelper.format("HH"));
        if (CollectionUtils.isNotEmpty(dorisAdGroupHourlyData)) {
            income = dorisAdGroupHourlyData.stream()
                    .map(DorisAdGroupHourlyData::getIncome)
                    .filter(Objects::nonNull)
                    .reduce(BigDecimal.ZERO, BigDecimal::add);
        }
        return income;
    }

    /**
     * 取max（配置的回传数）或lossThreshold(亏损阈值)的优先级：
     * 先取缓存中的、再取配置中的
     * 顺序都是，为空则继续取、直到不为空：
     * 1.指定adGroupId对应的数
     * 2.空adGroupId的数
     * 3.账户配置的数
     * @param adGroupId
     * @param positionRtb
     * @return
     */
    private FinalActiveParameter getFinalActiveParameter(String positionId, String adGroupId, PositionRtb positionRtb) {

        Integer max = fetchMaxFromCache(positionId, adGroupId);
        boolean useAccountDimension = false;

        // 取缓存的当日生效配置
        if (max == null) {
            max = fetchMaxFromCache(positionId, PreSummaryFeedbackConfig.EMPTY_FIELD);
            if (max == null) {
                max = fetchMaxFromCache(positionId, PreSummaryFeedbackConfig.ACCOUNT_FIELD);
                // 如果到这里取值不为空 说明需要使用账户维度数
                useAccountDimension = max != null;
            }
        }

        // 取最新配置的
        if (max == null) {
            PreSummaryFeedbackConfig preSummaryFeedbackConfig = getPreSummaryFeedbackConfig(positionRtb);
            if (preSummaryFeedbackConfig != null) {
                max = fetchMaxFromAdGroupConfigs(preSummaryFeedbackConfig.getAdGroupConfigs(), adGroupId);
                if (max == null) {
                    max = fetchMaxFromAdGroupConfigs(preSummaryFeedbackConfig.getAdGroupConfigs(), Strings.EMPTY);
                    if (max == null) {
                        max = preSummaryFeedbackConfig.getAccountMax();
                        // 如果到这里取值不为空 说明需要使用账户维度数
                        useAccountDimension = max != null;
                    }
                }
            }
        }

        BigDecimal lossThreshold = fetchLossThresholdFromCache(positionId, adGroupId);

        // 取缓存的当日生效配置
        if (lossThreshold == null) {
            lossThreshold = fetchLossThresholdFromCache(positionId, PreSummaryFeedbackConfig.EMPTY_FIELD);
            if (lossThreshold == null) {
                lossThreshold = fetchLossThresholdFromCache(positionId, PreSummaryFeedbackConfig.ACCOUNT_FIELD);
            }
        }

        // 取最新配置的
        if (lossThreshold == null) {
            PreSummaryFeedbackConfig preSummaryFeedbackConfig = getPreSummaryFeedbackConfig(positionRtb);
            if (preSummaryFeedbackConfig != null) {
                lossThreshold = fetchLossThresholdFromAdGroupConfigs(preSummaryFeedbackConfig.getAdGroupConfigs(), adGroupId);
                if (lossThreshold == null) {
                    lossThreshold = fetchLossThresholdFromAdGroupConfigs(preSummaryFeedbackConfig.getAdGroupConfigs(), Strings.EMPTY);
                    if (lossThreshold == null) {
                        lossThreshold = preSummaryFeedbackConfig.getAccountLossThreshold();
                    }
                }
            }
        }

        return new FinalActiveParameter()
                .setFeedbackMaxNum(max)
                .setLossThreshold(lossThreshold)
                .setUseAccountDimension(useAccountDimension);
    }

    private Integer fetchMaxFromCache(String positionId, String dimension) {
        return fetcher.hfetch(hashPreSummaryFeedbackConf(positionId), dimension, Integer.class);
    }

    private BigDecimal fetchLossThresholdFromCache(String positionId, String dimension) {
        return fetcher.hfetch(hashPreSummaryFeedbackLossThreshold(positionId), dimension, BigDecimal.class);
    }

    private Integer fetchMaxFromAdGroupConfigs(List<PreSummaryFeedbackConfig.AdGroupConfig> adGroupConfigs, String adGroupId) {
        return adGroupConfigs.stream()
                .filter(config -> Objects.equals(config.getAdGroupId(), adGroupId))
                .findFirst()
                .map(PreSummaryFeedbackConfig.AdGroupConfig::getMax)
                .orElse(null);
    }

    private BigDecimal fetchLossThresholdFromAdGroupConfigs(List<PreSummaryFeedbackConfig.AdGroupConfig> adGroupConfigs, String adGroupId) {
        return adGroupConfigs.stream()
                .filter(config -> Objects.equals(config.getAdGroupId(), adGroupId))
                .findFirst()
                .map(PreSummaryFeedbackConfig.AdGroupConfig::getLossThreshold)
                .orElse(null);
    }

    private PreSummaryFeedbackConfig getPreSummaryFeedbackConfig(PositionRtb positionRtb) {
        String preSummaryFeedbackConf = positionRtb.getPreSummaryFeedbackConf();
        if (StringUtils.isNotBlank(preSummaryFeedbackConf)) {
            return JsonHelper.convert(preSummaryFeedbackConf, PreSummaryFeedbackConfig.class);
        }
        return null;
    }

    public static KeyGenerator hashPreSummaryFeedbackConf(String positionId) {
        return () -> KeyBuilder.build("rtb", "pre_summary_feedback", "conf", getDate(), positionId);
    }

    public static KeyGenerator hashPreSummaryFeedbackCount(String positionId) {
        return () -> KeyBuilder.build("rtb", "pre_summary_feedback", "count", getDate(), positionId);
    }

    public static KeyGenerator hashPreSummaryFeedbackLossThreshold(String positionId) {
        return () -> KeyBuilder.build("rtb", "pre_summary_feedback", "loss_threshold", getDate(), positionId);
    }

    public static KeyGenerator hashPreSummaryFeedbackLossThresholdCount(String positionId) {
        return () -> KeyBuilder.build("rtb", "pre_summary_feedback", "loss_threshold_count", getDate(), positionId);
    }

    private static String getDate() {
        return LocalDate.now().format(DateTimeFormatter.ofPattern("yyyyMMdd"));
    }

    private BigDecimal computeProfit(BigDecimal income, BigDecimal charge) {
        BigDecimal profit = charge.subtract(income);
        log.info("computeProfit({}) = {} - {}", profit, charge, income);
        return profit;
    }


    @Data
    @Accessors(chain = true)
    private static class FinalActiveParameter {
        private Integer feedbackMaxNum;

        private BigDecimal lossThreshold;

        private boolean useAccountDimension;
    }

    private boolean isAccountDimension(String adGroupId) {
        return Objects.equals(adGroupId, "ACCOUNT");
    }

}
