package com.bxm.adscounter.rtb.common.control.cpa;

import com.bxm.adscounter.rtb.common.RtbIntegration;
import com.bxm.adscounter.rtb.common.RtbProperties;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.openlog.extension.client.OpenLogClient;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.Production;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.openlog.sdk.consts.Inads;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.utils.TypeHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.logging.log4j.util.Strings;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.time.Duration;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.*;

/**
 * @author tangxiao
 * @date 2023/6/8
 * @since 1.0
 */
@Slf4j
public class SummaryCpaControlImpl implements CpaControl{

    private static final int ONE_DAY_SEC = TypeHelper.castToInt(Duration.ofDays(1).getSeconds());
    private static final int MONTH_DAYS_SEC = TypeHelper.castToInt(Duration.ofDays(30).getSeconds());

    private final Counter counter;
    private final Fetcher fetcher;
    private final RtbProperties properties;
    private final OpenLogClient openLogClient;
    private final JedisPool jedisPool;
    private final PreSummaryFeedbackHelper preSummaryFeedbackHelper;

    public SummaryCpaControlImpl(Counter counter, Fetcher fetcher, RtbProperties properties, OpenLogClient openLogClient, JedisPool jedisPool, PreSummaryFeedbackHelper preSummaryFeedbackHelper) {
        this.counter = counter;
        this.fetcher = fetcher;
        this.properties = properties;
        this.openLogClient = openLogClient;
        this.jedisPool = jedisPool;
        this.preSummaryFeedbackHelper = preSummaryFeedbackHelper;
    }

    /**
     * X:累加的cpa，Y:后台配置的累加目标cpa
     * 当X>=Y, 则回传一次，并X=X-Y。
     *
     * 另外两个数据：
     * summaryCpa:累加cpa总金额，即每次累加当前cpa。
     * summaryCpaBalance:累加的cpa余额，即每次累加当前余额。
     * @param feedbackRequest
     * @param integration
     * @return true：回传，false：不回传
     */
    @Override
    public boolean isFeedback(FeedbackRequest feedbackRequest, RtbIntegration integration) {
        PositionRtb positionRtb = feedbackRequest.getConfig();
        KeyValueMap keyValueMap = feedbackRequest.getKeyValueMap();

        // 保持这个为前置判断，不是累加CPA模式不走下面流程
        if (!positionRtb.isFeedbackModeSummaryCpa()) {
            return true;
        }

        // CPA累加模式下的非有效转化不回传
        if (!isConversionValid(keyValueMap)) {
            return false;
        }

        Integer rtbTargetCpa = positionRtb.getRtbTargetCpa();
        if (rtbTargetCpa == null) {
            return true;
        }

        String positionId = positionRtb.getPositionId();
        String adGroupId = feedbackRequest.getAdGroupId();

        // 前置回传
        if (preFeedbackIfNeeded(positionRtb, positionId, adGroupId)) {
            return true;
        }

        int ocpaOfferPrice = NumberUtils.toInt(keyValueMap.getFirst(Inads.Param.OCPA_OFFER_PRICE));

        // 限制3倍
        if (!Objects.equals(rtbTargetCpa, 0)) {
            int limitOfferPrice = 3 * rtbTargetCpa;
            ocpaOfferPrice = Math.min(ocpaOfferPrice, limitOfferPrice);
        }

        KeyGenerator redisKey = hashRtbCpa(positionId, adGroupId);
        Long cpaBalance = counter.hincrementByAndGet(redisKey, Field.CPA_BALANCE, ocpaOfferPrice);

        boolean isFeedback = cpaBalance >= rtbTargetCpa;

        // 若回传扣除rtbTargetCpa
        if (isFeedback) {
            counter.hincrementByAndGet(redisKey, Field.CPA_BALANCE, -rtbTargetCpa);
        } else {
            // 暂存这次的回传
            pushConversion(feedbackRequest);
        }

        // 累加cpa出价
        counter.hincrementByAndGet(redisKey, Field.SUMMARY_CPA, ocpaOfferPrice);

        counter.expire(redisKey, ONE_DAY_SEC);

        sendOpenLog(positionId, adGroupId, positionRtb.getSourceType());

        return isFeedback;
    }

    @Override
    public List<FeedbackRequest> getOnConversionQueue(String positionId, List<String> adGroupIds, Integer size) {
        if (size <= 0) {
            return Collections.emptyList();
        }
        List<FeedbackRequest> results = new ArrayList<>();
        for (String adGroupId : adGroupIds) {
            String key = listConversionQueue(positionId, adGroupId).generateKey();
            do {
                String pop = leftPop(key);
                if (StringUtils.isBlank(pop)) {
                    break;
                }
                FeedbackRequest feedbackRequest = JsonHelper.convert(pop, FeedbackRequest.class);
                results.add(feedbackRequest);
            } while (results.size() < size);
        }
        return results;
    }

    @Override
    public void pushConversion(FeedbackRequest feedbackRequest) {
        PositionRtb config = feedbackRequest.getConfig();
        if(config.isEnableSmartHosting()) {
            String adGroupId = feedbackRequest.getAdGroupId();
            String positionId = config.getPositionId();
            String data = JsonHelper.convert(feedbackRequest);
            if (log.isInfoEnabled()) {
                log.info("累加回传暂存数据...{} - {}", positionId, adGroupId);
            }
            addConversion(positionId, adGroupId, data);
        }
    }

    @Override
    public long getConversionQueueLength(String positionId, List<String> adGroupIds) {
        long length = 0;
        for (String adGroupId : adGroupIds) {
            String key = listConversionQueue(positionId, adGroupId).generateKey();
            length += queueLength(key);
        }
        return length;
    }

    @Override
    public Long getSummaryCpaBalance(String positionId, String adGroupId) {
        KeyGenerator redisKey = hashRtbCpa(positionId, adGroupId);
        return Optional.ofNullable(fetcher.hfetch(redisKey, Field.CPA_BALANCE, Long.class)).orElse(0L);
    }

    /**
     * 是否直接回传
     * @param positionRtb
     * @param positionId
     * @param adGroupId
     * @return
     */
    private boolean preFeedbackIfNeeded(PositionRtb positionRtb, String positionId, String adGroupId) {
        // 判断累加前置回传配置
        if (positionRtb.isPreSummaryFeedback()) {
            return preSummaryFeedbackHelper.needFeedback(positionRtb, adGroupId);
        } else {
            // 否则默认首次直接回传
            boolean isInitialized = counter.incrementAndGet(strInitSummary(positionId, adGroupId), MONTH_DAYS_SEC) > 1;
            return !isInitialized;
        }
    }

    public KeyGenerator strInitSummary(String positionId, String adGroupId) {
        return () -> KeyBuilder.build("rtb", "cpa_init", positionId, adGroupId);
    }

    public KeyGenerator hashRtbCpa(String positionId, String adGroupId) {
        return () -> KeyBuilder.build("rtb", "cpa", positionId, adGroupId, getDate());
    }

    private static String getDate() {
        return LocalDate.now().format(DateTimeFormatter.ofPattern("yyyyMMdd"));
    }

    interface Field {
        /**
         * 当前余额，相当于回传的额度 (累加CPA余额)
         */
        String CPA_BALANCE = "cpa_balance";

        /**
         * 累加CPA
         */
        String SUMMARY_CPA = "summary_cpa";
    }

    private void sendOpenLog(String positionId, String adGroupId, Integer rtbId) {
        Map<String, Long> rtbCpaData = fetcher.hfetchall(hashRtbCpa(positionId, adGroupId), Long.class);
        Long summaryCpa = Optional.ofNullable(rtbCpaData.get(Field.SUMMARY_CPA)).orElse(0L);
        Long cpaBalance = Optional.ofNullable(rtbCpaData.get(Field.CPA_BALANCE)).orElse(0L);

        KeyValueMap map = new KeyValueMap();

        map.put(Common.RtbParam.P, Production.COMMON.getName());
        map.put(Common.RtbParam.MT, Common.Mt.RtbSummaryCpa.original());
        map.put(Common.RtbParam.TIME, System.currentTimeMillis());
        map.put(Common.RtbParam.TAGID, positionId);
        map.put(Common.Param.RTB_ID, rtbId);
        map.put(Common.Param.AD_GROUP_ID, adGroupId);
        map.put(Field.SUMMARY_CPA, summaryCpa);
        map.put("summary_cpa_balance", cpaBalance);

        String openLogRequestUri = map.createOpenLogRequestUri(properties.getOpenLogRequestDomain());
        openLogClient.asyncRequest(openLogRequestUri);
    }

    public boolean isConversionValid(KeyValueMap map) {
        return StringUtils.equalsIgnoreCase(map.getFirst(Inads.Param.CONVERSION_VALID), "1");
    }

    static KeyGenerator listConversionQueue(String positionId, String adGroupId) {
        return () -> KeyBuilder.build("rtb", "summary_feedback", "CONV_QUEUE", getHour(), positionId, adGroupId);
    }

    private static String getHour() {
        return DateHelper.format("yyyyMMddHH");
    }

    private void addConversion(String positionId, String adGroupId, String data) {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = listConversionQueue(positionId, adGroupId).generateKey();
            jedis.lpush(key, data);
            jedis.expire(key, 24 * 60 * 60);
        }
    }

    public JedisPool getJedisPool() {
        return jedisPool;
    }

    private String leftPop(String key) {
        try (Jedis jedis = getJedisPool().getResource()) {
            return jedis.lpop(key);
        } catch (Exception e) {
            log.error("lpop: {}", e.getMessage());
            return null;
        }
    }

    private long queueLength(String key) {
        try (Jedis jedis = getJedisPool().getResource()) {
            return jedis.llen(key);
        } catch (Exception e) {
            log.error("lpop: {}", e.getMessage());
            return 0;
        }
    }
}
