package com.bxm.adscounter.rtb.common.control.deduction;

import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.utils.TypeHelper;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Tag;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.time.Duration;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;

/**
 * 基于 Redis 实现的比例控制器实现
 *
 * @author tangxiao
 * @date 2022-08-05
 * @since 1.0
 * @see com.bxm.adscounter.rtb.common.control.ratio.RedisRatioControlImpl
 */
@Slf4j
@Deprecated
public class RedisDeductionControlImpl implements DeductionControl {

    private static final int ONE_DAY_SEC = TypeHelper.castToInt(Duration.ofDays(1).getSeconds());

    private final DeductionControlConfig config;
    private final JedisPool jedisPool;
    private final MeterRegistry registry;

    public RedisDeductionControlImpl(JedisPool jedisPool, DeductionControlConfig config, MeterRegistry registry) {
        this.registry = registry;
        this.jedisPool = jedisPool;
        this.config = config;

        Preconditions.checkNotNull(config);
        Preconditions.checkArgument(StringUtils.isNotBlank(config.getDimension()), "dimension cannot be null");
        Preconditions.checkNotNull(config.getConsumer(), "consumer cannot be null");
        double ratio = config.getRatio();
        Preconditions.checkArgument(ratio >= 0 && ratio <= 1, "ratio must >= 0 and <= 1");
    }

    @Override
    public void accept() {
        this.countAccept();
        this.countAcceptForGlobal();
        this.registry.counter("deduction.control.accept", tags()).increment();
    }

    @Override
    public void reject(String data) {
        this.countReject();
        this.countRejectForGlobal();
        Consumer<String> deductionConsumer = config.getDeductionConsumer();

        if (Objects.nonNull(deductionConsumer)) {
            deductionConsumer.accept(data);
        }

        this.registry.counter("deduction.control.reject", tags()).increment();
    }

    @Override
    public void push(String data) {
        double stepValue = 1d;
        double zeroRatio = 0d;

        if (!this.isInitializedCountRate() && config.getRatio() > zeroRatio) {
            this.incrCountRateBy(stepValue);
        }

        double after = this.incrCountRateBy(config.getRatio());
        boolean feedbackIfNecessary = after >= stepValue;
        if (feedbackIfNecessary) {
            this.incrCountRateBy( - stepValue);
            this.accept();

            Consumer<String> consumer = config.getConsumer();
            try {
                if (Objects.nonNull(consumer)) {
                    consumer.accept(data);
                }
            } catch (Exception e) {
                log.error("occur exception | accept: ", e);
            }
        } else {
            this.reject(data);
        }

        this.registry.counter("deduction.control", tags()).increment();
    }

    @Override
    public void delete() {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String dimension = config.getDimension();
            String stringRateCountInitialized = stringRateCountInitialized(dimension).generateKey();
            String stringCountRate = stringCountRate(dimension).generateKey();
            String stringAccept = stringAccept(dimension).generateKey();
            String stringAcceptForGlobal = stringAcceptForGlobal(Objects.toString(config.getHitConfigId())).generateKey();
            String stringReject = stringReject(dimension).generateKey();
            String stringRejectForGlobal = stringRejectForGlobal(Objects.toString(config.getHitConfigId())).generateKey();
            jedis.del(stringRateCountInitialized, stringCountRate, stringAccept, stringAcceptForGlobal, stringReject, stringRejectForGlobal);
        }
    }

    @Override
    public DeductionControlConfig getConfig() {
        return this.config;
    }

    private List<Tag> tags() {
        return Lists.newArrayList(Tag.of("dim", config.getDimension()));
    }

    private JedisPool getJedisPool() {
        return this.jedisPool;
    }

    // ----------- redis operation functions ----------- //

    private boolean isInitializedCountRate() {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = stringRateCountInitialized(config.getDimension()).generateKey();
            Long rs = jedis.incr(key);
            jedis.expire(key, ONE_DAY_SEC);
            if (log.isDebugEnabled()) {
                log.debug("{} after incr {}", key, rs);
            }
            return Optional.ofNullable(rs).orElse(1L) > 1;
        }
    }

    private double incrCountRateBy(double incrementValue) {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = stringCountRate(config.getDimension()).generateKey();
            double rs = jedis.incrByFloat(key, incrementValue);
            jedis.expire(key, ONE_DAY_SEC);
            if (log.isDebugEnabled()) {
                log.debug("{} after incrBy {}", key, rs);
            }
            return rs;
        }
    }

    private void countAccept() {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = stringAccept(config.getDimension()).generateKey();
            Long rs = jedis.incr(key);
            jedis.expire(key, ONE_DAY_SEC);
            if (log.isDebugEnabled()) {
                log.debug("{} after incrBy {}", key, rs);
            }
        }
    }

    private void countReject() {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = stringReject(config.getDimension()).generateKey();
            Long rs = jedis.incr(key);
            jedis.expire(key, ONE_DAY_SEC);
            if (log.isDebugEnabled()) {
                log.debug("{} after incrBy {}", key, rs);
            }
        }
    }

    private void countAcceptForGlobal() {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = stringAcceptForGlobal(Objects.toString(config.getHitConfigId())).generateKey();
            Long rs = jedis.incr(key);
            jedis.expire(key, ONE_DAY_SEC);
            if (log.isDebugEnabled()) {
                log.debug("{} after incrBy {}", key, rs);
            }
        }
    }

    private void countRejectForGlobal() {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = stringRejectForGlobal(Objects.toString(config.getHitConfigId())).generateKey();
            Long rs = jedis.incr(key);
            jedis.expire(key, ONE_DAY_SEC);
            if (log.isDebugEnabled()) {
                log.debug("{} after incrBy {}", key, rs);
            }
        }
    }

    // ----------- common static functions ----------- //

    private static KeyGenerator stringRateCountInitialized(String dimension) {
        return () -> KeyBuilder.build("rtb_control", "deduction", "init", getDate(), dimension);
    }

    private static KeyGenerator stringCountRate(String dimension) {
        return () -> KeyBuilder.build("rtb_control", "deduction", "rate", getDate(), dimension);
    }

    private static KeyGenerator stringAccept(String dimension) {
        return () -> KeyBuilder.build("rtb_control", "deduction", "accept", getDate(), dimension);
    }

    private static KeyGenerator stringAcceptForGlobal(String dimension) {
        return () -> KeyBuilder.build("rtb_control", "deduction", "accept_global", getDate(), dimension);
    }

    private static KeyGenerator stringReject(String dimension) {
        return () -> KeyBuilder.build("rtb_control", "deduction", "reject", getDate(), dimension);
    }

    private static KeyGenerator stringRejectForGlobal(String dimension) {
        return () -> KeyBuilder.build("rtb_control", "deduction", "reject_global", getDate(), dimension);
    }

    private static String getDate() {
        return LocalDate.now().format(DateTimeFormatter.ofPattern("yyyyMMdd"));
    }
}
