package com.bxm.adscounter.rtb.common.control.rate;

import com.alibaba.fastjson.JSONObject;
import com.bxm.adscounter.rtb.common.control.AbstractStandaloneControlScheduler;
import com.bxm.adscounter.rtb.common.data.QueryParam;
import com.bxm.adscounter.rtb.common.data.SrcAdUserAccessLog;
import com.bxm.adscounter.rtb.common.feedback.ActionType;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.SmartConvType;
import com.bxm.adscounter.rtb.common.mapper.SrcAdUserAccessLogMapper;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Inads;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.JsonHelper;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Timer;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import redis.clients.jedis.JedisPool;

import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * 基于 Redis 的速率控制器调度器。
 *
 * @author allen
 * @date 2022-07-13
 * @since 1.0
 */
@Slf4j
public class RedisRateControlScheduler extends AbstractStandaloneControlScheduler {

    private final RateControl control;

    private final SrcAdUserAccessLogMapper srcAdUserAccessLogMapper;

    private final MeterRegistry registry;

    public RedisRateControlScheduler(RateControl control, JedisPool jedisPool, EventPark eventPark, SrcAdUserAccessLogMapper srcAdUserAccessLogMapper, MeterRegistry registry) {
        super(null, jedisPool, eventPark);
        this.control = control;
        this.srcAdUserAccessLogMapper = srcAdUserAccessLogMapper;
        this.registry = registry;
    }

    @Override
    protected void doRun() {
        RateControlConfig config = control.getConfig();
        PositionRtb positionRtbConfig = config.getPositionRtb();

        long current = control.getCount();
        long total = config.getCount();
        if (current >= total) {
            // 额度已满
            return;
        }

        FeedbackRequest lastOne = getOneFeedbackRequestFromDB(config, positionRtbConfig);

        if (Objects.nonNull(lastOne)) {
            try {
                config.getConsumer().accept(lastOne);
            } catch (Exception e) {
                log.error("occur exception | accept: ", e);
            } finally {
                control.count();
                if (log.isDebugEnabled()) {
                    log.debug("[{}] progress: {}/{}", config.getDimension(), control.getCount(), total);
                }
            }
        }
    }

    private FeedbackRequest getOneFeedbackRequestFromDB(RateControlConfig config, PositionRtb positionRtbConfig) {
        Timer timer = registry.timer("rtb.querydb.cost");
        long start = System.currentTimeMillis();
        try {
            String positionId = config.getPositionId();
            if (StringUtils.isBlank(positionId)) {
                return null;
            }
            String[] split = positionId.split("-");
            String appKey = split[0];
            String business = split[1];

            QueryParam queryParam = QueryParam.builder()
                    .appKey(appKey)
                    .business("money-" + business)
                    .date(DateHelper.format(DateHelper.PATTERN_STR8))
                    .ad_group_id(config.getAdGroupId())
                    .action(getAction(config.getActionType()))
                    .adId(config.getAdId())
                    .build();

            SrcAdUserAccessLog srcAdUserAccessLog = srcAdUserAccessLogMapper.getLastOne(queryParam);

            if (srcAdUserAccessLog != null) {
                log.info("rate control found log. {}", srcAdUserAccessLog.toSimpleString());
                return of(srcAdUserAccessLog, positionRtbConfig, config);
            }

            log.info("Cannot found log from database. {}", queryParam.toSimpleString());
        } catch (Exception e) {
            log.error("getOneFeedbackRequestFromDB occur error", e);
        } finally {
            timer.record((System.currentTimeMillis() - start), TimeUnit.MILLISECONDS);
        }
        return null;
    }

    public FeedbackRequest of(SrcAdUserAccessLog log, PositionRtb positionRtbConfig, RateControlConfig rateControlConfig) {
        String rtbExt = log.getRtbExt();

        String clickId = null;
        if (StringUtils.isNotBlank(rtbExt)) {
            JSONObject rtbExtJsonObject = JsonHelper.convert(rtbExt, JSONObject.class);
            clickId = rtbExtJsonObject.getString("click_id");
        }

        KeyValueMap keyValueMap = new KeyValueMap();
        keyValueMap.put(Inads.Param.OAID, log.getOaid());
        keyValueMap.put(Inads.Param.OAID_MD5, log.getOaidMd5());
        keyValueMap.put(Inads.Param.IMEI, log.getImei());
        keyValueMap.put(Inads.Param.IMEI_MD5, log.getImeiMd5());
        keyValueMap.put(Inads.Param.IDFA, log.getIdfa());
        keyValueMap.put(Inads.Param.IDFA_MD5, log.getIdfaMd5());
        keyValueMap.put(Inads.Param.ANDROIDID, log.getAnid());
        keyValueMap.put(Inads.Param.ANDROIDID_MD5, log.getAnidMd5());
        keyValueMap.put(Inads.Param.BXMID, log.getBxmId());
        keyValueMap.put(Inads.Param.IP, log.getIp());
        keyValueMap.put(Inads.Param.UA, log.getUas());
        keyValueMap.put(Inads.Param.UID, log.getUid());
        keyValueMap.put(Inads.Param.TIME, log.getTimeStamp());
        keyValueMap.put(Inads.Param.ADID, log.getPreId());
        keyValueMap.put(Inads.Param.TAGID, positionRtbConfig.getPositionId());

        return FeedbackRequest
                .builder()
                .config(positionRtbConfig)
                .conversionLevel(FeedbackRequest.SHALLOW_CONVERSION_LEVEL)
                .conversionType("0")
                .smartConvType(SmartConvType.NONE)
                .referrer(log.getRefer())
                .adGroupId(rateControlConfig.getAdGroupId())
                .clickId(clickId)
                .eventType(positionRtbConfig.getTargetOneRtb())
                .keyValueMap(keyValueMap)
                .actionType(rateControlConfig.getActionType())
                .build();
    }

    private Integer getAction(ActionType actionType) {
        return ActionType.TICKET_CLICK.equals(actionType) || ActionType.TICKET_CONVERSION.equals(actionType) ? 2 : 1;
    }
}
