package com.bxm.adscounter.rtb.common.control.ratio;

import com.bxm.adscounter.rtb.common.DataFetchFailException;
import com.bxm.adscounter.rtb.common.DataFetcher;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegration;
import com.bxm.adscounter.rtb.common.control.ratio.event.AdGroupFetchEvent;
import com.bxm.adscounter.rtb.common.control.ratio.event.AdGroupFetchExceptionEvent;
import com.bxm.adscounter.rtb.common.data.AdGroupData;
import com.bxm.adscounter.rtb.common.data.Parameter;
import com.bxm.warcar.integration.eventbus.EventPark;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;

import java.util.List;
import java.util.Objects;

/**
 * @author allen
 * @date 2022-11-15
 * @since 1.0
 */
@Slf4j
public class DataFetchingScheduler implements Runnable {

    private final RatioControl control;
    private final RatioControlBus bus;

    public DataFetchingScheduler(RatioControl control) {
        this.control = control;
        this.bus = control.getBus();
    }

    @Override
    public void run() {
        RatioControlConfig config = getConfig();
        EventPark eventPark = bus.getEventPark();
        try {
            if (log.isDebugEnabled()) {
                log.debug("[{}] Starting execute", config.getDimension());
            }
            Rtb rtb = bus.getInstance().rtb();
            DataFetcher fetcher = bus.getFactory().getDataFetcher(rtb);
            if (Objects.isNull(fetcher)) {
                return;
            }
            if (!config.isEnableCostControl()) {
                // 如果没有开启效果成本控制功能就不需要获取数据了。
                return;
            }

            Parameter parameter = Parameter.builder()
                    .advertiserId(config.getAdvertiserId())
                    .tagId(config.getTagId())
                    .adGroupId(config.getAdGroupId())
                    .shallowEventType(config.getShallowEventType())
                    .deepEventType(config.getDeepEventType())
                    .advertiserName(config.getAdvertiserName())
                    .build();

            List<AdGroupData> adGroupData = fetcher.fetchCurrentHourData(parameter);
            if (CollectionUtils.isEmpty(adGroupData)) {
                log.info("[{}] Not fetch data for ad_group_id={}", config.getDimension(), parameter.getAdGroupId());
                return;
            }

            eventPark.post(new AdGroupFetchEvent(this, control, adGroupData, true));
            if (log.isInfoEnabled()) {
                log.info("[{}] Finish fetch task, size: {}, result: {}", config.getDimension(), adGroupData.size(), adGroupData);
            }

            this.processPreviousHourData(fetcher, parameter);

        } catch (DataFetchFailException e) {
            eventPark.post(new AdGroupFetchExceptionEvent(this, e));
        } catch (Exception e) {
            log.error("run: ", e);
        }
    }

    /**
     * 处理上一个小时的数据，获取到数据集合后发送给OpenLog，数据中心会对小时数据进行更新。
     * 以修正实时数据在最后几分钟因没有及时刷新导致结果不正确的问题。
     *
     * @param fetcher Fetcher
     * @param parameter 参数
     */
    private void processPreviousHourData(DataFetcher fetcher, Parameter parameter) {
        try {
            List<AdGroupData> previousHour = fetcher.fetchPreviousHourData(parameter);
            if (CollectionUtils.isNotEmpty(previousHour)) {
                bus.getEventPark().post(new AdGroupFetchEvent(this, control, previousHour, false));
            }
        } catch (DataFetchFailException e) {
            log.warn("fetchPreviousHourData: {}", e.getMessage());
        }
    }

    private RatioControlConfig getConfig() {
        return this.control.getConfig();
    }
}
