package com.bxm.adscounter.rtb.common.control.ratio;

import com.bxm.adscounter.rtb.common.control.deduction.ConversionLevel;
import lombok.Data;
import lombok.experimental.Accessors;

import java.time.Duration;
import java.util.Objects;

/**
 * 调控配置。
 *
 * @author allen
 * @date 2022-07-11
 * @since 1.0
 */
@Data
@Accessors(chain = true)
public class RatioControlConfig {

    /**
     * <p>广告位ID</p>
     */
    private String tagId;

    /**
     * <p>RTB advertiserId</p>
     */
    private String advertiserId;

    /**
     * RTB advertiser名称
     */
    private String advertiserName;

    /**
     * <p>ad_group_id</p>
     */
    private String adGroupId;

    /**
     * <p>维度，全局唯一。一般可以是广告ID、广告位等。</p>
     */
    private String dimension;

    /**
     * 当前命中控制配置的id
     */
    private Long hitConfigId;

    /**
     * <p>CVR 期望的比率。1 表示 100%。</p>
     */
    private double ratio = 1;

    /**
     * <p>浅层转化扣量的比例。1 表示 100%。</p>
     */
    private double deductionShallowRatio = 1;

    /**
     * <p>深层转化扣量的比例。1 表示 100%。</p>
     */
    private double deductionDeepRatio = 1;

    /**
     * <p>期望的效果成本</p>
     */
    private double cost;

    /**
     * <p>比率控制执行间隔时间，默认：30 秒。</p>
     * <p>设置时间越长，对于比率的波动越大；时间越短，对整体性能影响越大。</p>
     */
    private int intervalInSec = 30;

    /**
     * <p>每次获取数据执行间隔时间，默认：10 分钟</p>
     */
    private int dataFetchInMinute = 10;

    /**
     * <p>转化队列过期时间，单位：小时，默认：24 小时。</p>
     * <p>设置时间越长，占用内存空间越大；时间越短，有效性越差。</p>
     */
    private int expireInHours = 24;

    /**
     * <p>获取到的数据过期时间，单位：分钟，默认：10 分钟。</p>
     */
    private int dataExpireInMinute = 10;

    /**
     * <p>开启需要的最低转化数</p>
     */
    private long onsetOfConversion = 20;

    /**
     * <p>是否开启了CVR控制</p>
     */
    private boolean enableCvrControl;

    /**
     * <p>是否开启了扣量控制</p>
     */
    private boolean enableDeductionControl;

    /**
     * <p>是否开启了效果成本控制</p>
     */
    private boolean enableCostControl;

    /**
     * <p>回传 RTB 浅层转化目标</p>
     */
    private String shallowEventType;

    /**
     * <p>回传 RTB 深层转化目标</p>
     */
    private String deepEventType;

    /**
     * <p>回传 RTB 所配置的目标类型</p>
     */
    private ConversionLevel conversionLevel;

    /**
     * <p>限制非转化回传的数量，如果为空表示不限</p>
     */
    private Integer limitUnconvs;

    /**
     * 取数时间
     */
    private Duration2 duration = Duration2.DAY;

    /**
     * 补量策略
     */
    private PlusStrategy plusStrategy = PlusStrategy.CONV_COUNT;

    public int getExpireInSeconds() {
        return (int) Duration.ofHours(expireInHours).getSeconds();
    }

    public int getDataExpireInSeconds() {
        return (int) Duration.ofMinutes(dataExpireInMinute).getSeconds();
    }

    /**
     * 获取本次可补的非转化数量。
     * 如果设置不限，那么一定返回 {@code remain} 次；否则限制数量减去已经补量数，与 {@code remain} 取小。
     *
     * @param current 当前已经补量数
     * @param remain 本次需要补量数
     * @return 本次可补的数量
     */
    public int minForLimit(long current, int remain) {
        if (Objects.isNull(limitUnconvs)) {
            return remain;
        }
        int a = limitUnconvs - (int) current;
        return Math.min(a, remain);
    }

    public enum Duration2 {
        /**
         * 按天
         */
        DAY(1),
        /**
         * 按小时
         */
        HOUR(2);

        private final int value;

        Duration2(int value) {
            this.value = value;
        }

        public int getValue() {
            return value;
        }

        public static Duration2 of(Integer v) {
            if (Objects.isNull(v)) {
                return null;
            }
            for (Duration2 d : values()) {
                if (d.getValue() == v) {
                    return d;
                }
            }
            return null;
        }
    }

    public enum PlusStrategy {
        /**
         * 转化数
         */
        CONV_COUNT(1),
        /**
         * CVR
         */
        CVR(2);

        private final int value;

        PlusStrategy(int value) {
            this.value = value;
        }

        public int getValue() {
            return value;
        }

        public static PlusStrategy of(Integer v) {
            if (Objects.isNull(v)) {
                return null;
            }
            for (PlusStrategy d : values()) {
                if (d.getValue() == v) {
                    return d;
                }
            }
            return null;
        }
    }
}
