package com.bxm.adscounter.rtb.common.control.ratio;

import com.bxm.adscounter.rtb.common.ClickTracker;
import com.bxm.adscounter.rtb.common.RtbIntegration;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.control.AbstractStandaloneControlScheduler;
import com.bxm.adscounter.rtb.common.control.ControlUtils;
import com.bxm.adscounter.rtb.common.control.LocalDateTimeUtils;
import com.bxm.adscounter.rtb.common.control.deduction.ConversionLevel;
import com.bxm.adscounter.rtb.common.control.ratio.event.RatioFeedbackEvent;
import com.bxm.adscounter.rtb.common.control.ratio.event.RatioFeedbackExceptionEvent;
import com.bxm.adscounter.rtb.common.control.ratio.event.RatioPlusEvent;
import com.bxm.adscounter.rtb.common.data.AdGroupData;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.feedback.SmartConvType;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Inads;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.utils.JsonHelper;
import com.google.common.collect.Lists;
import lombok.Data;
import lombok.experimental.Accessors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import redis.clients.jedis.Jedis;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import static com.bxm.adscounter.rtb.common.control.ratio.RedisRatioControlImpl.*;

/**
 * 基于 Redis 的比率控制器调度器。
 *
 * @author allen
 * @date 2022-07-13
 * @since 1.0
 */
@Slf4j
public class RedisRatioControlScheduler extends AbstractStandaloneControlScheduler {

    private final RatioControl control;

    RedisRatioControlScheduler(RatioControl control) {
        super(control.getBus().getInstance(), control.getBus().getJedisPool(), control.getBus().getEventPark());
        this.control = control;
    }

    @Override
    protected void doRun() {
        RatioControlConfig config = control.getConfig();
        String dimension = getDimension();

        boolean enableDeductionControl = config.isEnableDeductionControl();
        boolean enableCvrControl = config.isEnableCvrControl();
        boolean enableCostControl = config.isEnableCostControl();
        if (!enableCvrControl && !enableCostControl) {
            // 如果没有开启CVR控制或者效果成本控制，就不需要执行了。
            return;
        }

        List<Conversion> result = Lists.newArrayList();
        RtbIntegration instance = getInstance();

        // 如果开启了效果成本控制，优先按这个处理，不管CVR控制有没有开启
        if (enableCostControl)
        {
            // dataList 必须保证在 X 过期前过期，并且 dataList 刷新后在下一次调度执行前必须保证 X 已重置。
            List<AdGroupData> dataList = control.getDataList();
            if (CollectionUtils.isEmpty(dataList)) {
                // 数据过期了，不走效果成本控制
                enableCostControl = false;
            } else {
                Stat stat = control.getStat();
                // 判断深浅配置
                int count = computeCostCount(config, dataList, stat.getX());
                result.addAll(getOnConversionAndClickQueue(stat, count));
            }
        }

        // 如果没有开启效果成本控制，但是开启了 CVR 控制，走 CVR 控制逻辑
        if (!enableCostControl && enableCvrControl)
        {
            Stat stat = control.getStat();

            long clickTotal = stat.getClicks();
            long conversions = stat.getConversions();

            if (log.isDebugEnabled()) {
                log.debug("[{}] Stat: {}", dimension, stat);
            }

            // 没有达到控制条件
            if (conversions < config.getOnsetOfConversion()) {
                List<Conversion> queue = getOnAppConversionQueueOrderBy(Integer.MAX_VALUE);
                if (CollectionUtils.isNotEmpty(queue)) {
                    result.addAll(queue);

                    if (log.isDebugEnabled()) {
                        log.debug("[{}] Starting progress {}/{}", dimension, conversions, config.getOnsetOfConversion());
                    }
                }
            } else {
                int count = computeCvrCount(config, clickTotal, conversions);

                // 没有开启扣量
                if (!enableDeductionControl) {
                    List<Conversion> conversionQueue = getOnAppConversionQueueOrderBy(Integer.MAX_VALUE);
                    result.addAll(conversionQueue);
                    count -= conversionQueue.size();
                }

                result.addAll(getOnConversionAndClickQueue(stat, count));
            }
        }

        // 如果都没有开启
        if (!enableCvrControl && !enableCostControl && !enableDeductionControl) {
            List<Conversion> conversions = getOnAppConversionQueueOrderBy(Integer.MAX_VALUE);
            result.addAll(conversions);
        }

        this.doConversion(result, instance, enableCostControl);

        if (log.isDebugEnabled()) {
            log.debug("[{}] Finished count: {}", dimension, result.size());
        }
    }

    private RatioControlConfig getConfig() {
        return this.control.getConfig();
    }

    private String getDimension() {
        return getConfig().getDimension();
    }

    /**
     * 根据 RTB 广告组数据计算当前需要回传的转化数量
     *
     * @param config 配置
     * @param dataList 数据结果集
     * @return 需要回传的转化数量
     */
    private int computeCostCount(RatioControlConfig config, List<AdGroupData> dataList, long x) {
        AdGroupData data = dataList.get(0);

        double cfgCost = config.getCost();

        BigDecimal charge = Optional.ofNullable(data.getCharge()).orElse(BigDecimal.ZERO);
        long rtbCount = Optional.ofNullable(data.getConvNumByImpression()).orElse(0L);

        if (config.getDuration() == RatioControlConfig.Duration2.DAY) {
            // 如果是「天」那么取对应的转化数
            if (config.getConversionLevel() == ConversionLevel.SHALLOW) {
                rtbCount = Optional.ofNullable(data.getShallowConvCount()).orElse(0L);
            } else if (config.getConversionLevel() == ConversionLevel.DEEP) {
                rtbCount = Optional.ofNullable(data.getDeepConvCount()).orElse(0L);
            }
        }

        BigDecimal g = charge.divide(BigDecimal.valueOf(cfgCost), 2, RoundingMode.HALF_UP)
                             .subtract(BigDecimal.valueOf(rtbCount))
                             .subtract(BigDecimal.valueOf(x));
        int count = g.intValue();

        if (log.isInfoEnabled()) {
            log.info("COST: (duration={},level={}): [{}] {}={}/{}-{}-{}", getDimension(),
                    config.getDuration(), config.getConversionLevel(),
                    g, charge, cfgCost, rtbCount, x);
        }
        return count;
    }

    /**
     * 根据当前的累计点击数、已回传的转化数，计算当前还需要回传的转化数量
     *
     * @param config 配置
     * @param clickTotal 累计点击数
     * @param conversions 已回传的转化数
     * @return 需要回传的转化数量
     */
    private int computeCvrCount(RatioControlConfig config, long clickTotal, long conversions) {
        double exceptRatio = config.getRatio();
        BigDecimal necessaryTotal = BigDecimal.valueOf(clickTotal).multiply(BigDecimal.valueOf(exceptRatio));
        int count = necessaryTotal.subtract(BigDecimal.valueOf(conversions)).intValue();
        if (log.isDebugEnabled()) {
            log.debug("CVR: [{}] {}={}*({}%)-{}", getDimension(), count, clickTotal, exceptRatio * 100, conversions);
        }
        return count;
    }

    private void doConversion(List<Conversion> result, RtbIntegration instance, boolean enableCostControl) {
        EventPark eventPark = getEventPark();

        // 执行回传逻辑
        for (Conversion conversion: result) {
            FeedbackRequest feedbackRequest = conversion.getRequest();
            if (Objects.isNull(feedbackRequest)) {
                continue;
            }
            String adGroupId = feedbackRequest.getAdGroupId();
            String appid = feedbackRequest.getAppid();
            ClickLog clickLog = conversion.getClickLog();

            try {
                FeedbackResponse response = instance.doFeedback(feedbackRequest, 1);
                if (response.isSuccess()) {
                    control.onConversion();
                    if (enableCostControl) {
                        control.onX();
                    }
                    eventPark.post(new RatioFeedbackEvent(this, control, adGroupId, appid, clickLog));
                    eventPark.post(new RatioPlusEvent(this, control, adGroupId, appid, clickLog));
                } else {
                    throw new RtbIntegrationException(null);
                }
            } catch (RtbIntegrationException ignore) {
                eventPark.post(new RatioFeedbackExceptionEvent(this, control, adGroupId, appid, clickLog));
            } catch (Exception e) {
                log.error("occur exception | accept: ", e);
            }
        }
    }

    private List<Conversion> convert(List<String> json) {
        if (CollectionUtils.isEmpty(json)) {
            return Lists.newArrayList();
        }
        return json.stream().map(this::convert).collect(Collectors.toList());
    }

    private Conversion convert(String json) {
        FeedbackRequest request = JsonHelper.convert(json, FeedbackRequest.class);
        request.setSmartConvType(SmartConvType.CONV_QUEUE);
        return new Conversion().setRequest(request);
    }

    private List<Conversion> getOnConversionAndClickQueue(Stat stat, int count) {
        List<Conversion> result = Lists.newArrayList();
        try {
            if (count > 0) {
                List<Conversion> conversionQueue = getOnAppConversionQueueOrderBy(count);
                result.addAll(conversionQueue);

                // 如果不足，从广告券转化队列中获取
                List<Conversion> ticketConversionQueue = Lists.newArrayList();
                int remain = count - result.size();
                if (remain > 0) {
                    ticketConversionQueue = getOnTicketConversionQueueOrderBy(remain);
                    result.addAll(ticketConversionQueue);
                }

                // 如果再不足，从点击里获取
                List<Conversion> clickQueue = Lists.newArrayList();
                remain = count - result.size();
                if (remain > 0) {
                    // 判断是否有实际转化数
                    if (stat.getReceivers() == 0) {
                        throw new IllegalStateException("没有实际转化数，所以不能进行补量。");
                    }
                    // 判断点击是否已经达到限额
                    remain = getConfig().minForLimit(stat.getPlus(), remain);
                    if (remain <= 0) {
                        throw new IllegalStateException(String.format("当前已补量数 %s 超过了限制数 %s。", stat.getPlus(), getConfig().getLimitUnconvs()));
                    }
                    clickQueue = getOnClickQueue(remain);
                    result.addAll(clickQueue);
                }

                if (log.isDebugEnabled()) {
                    log.debug("[{}] Need size: {} From Conversion={}, TicketConversion={}, Click={}", getDimension(), count, conversionQueue.size(), ticketConversionQueue.size(), clickQueue.size());
                }
            }
        } catch (IllegalStateException e) {
            if (log.isInfoEnabled()) {
                log.info("[{}] - {}", getDimension(), e.getMessage());
            }
        }
        return result;
    }

    private List<Conversion> getOnAppConversionQueueOrderBy(int size) {
        RatioControlConfig config = getConfig();
        String dimension = getDimension();

        int offset = 0;
        int step = 10;

        List<String> result = Lists.newArrayList();
        FINISH:
        do {
            int start = offset * step;
            int stop = (offset + 1) * step - 1;
            Set<String> topN = control.getTopN(start, stop);
            if (CollectionUtils.isEmpty(topN)) {
                if (log.isWarnEnabled()) {
                    log.warn("[{}] 没有策略排行单在转化时 {}-{}", config.getDimension(), start, stop);
                }
                break;
            }

            for (String app : topN) {
                String key = listAppConversionQueue(config.getDuration(), config.getDimension(), app).generateKey();
                String json;
                do {
                    json = rightPopOneOnKeyQueue(key);
                    if (StringUtils.isBlank(json)) {
                        break;
                    }
                    // 校验有效性
                    Conversion conversion = convert(json);
                    if (!isValid(conversion)) {
                        log.info("[{}] - 无效的转化：{}", dimension, conversion.getRequest().getKeyValueMap());
                        continue;
                    }

                    result.add(json);
                } while (result.size() < size);

                if (result.size() >= size) {
                    continue FINISH;
                }
                offset++;
            }
        } while (result.size() < size);

        return convert(result);
    }

    private List<Conversion> getOnTicketConversionQueueOrderBy(int size) {
        RatioControlConfig config = getConfig();
        // 这里的 dimension 字符串如果调整，需要同步修改下面这些地方：
        // com.bxm.adscounter.rtb.common.control.ticket.RedisTicketRatioControlImpl.pushConversion
        String ticketConversionDimension = ControlUtils.createKey(config.getTagId(), config.getAdGroupId());
        List<String> result = Lists.newArrayList();

        do {
            String key = listTicketConversionQueue(ticketConversionDimension).generateKey();
            String json = rightPopOneOnKeyQueue(key);
            if (StringUtils.isBlank(json)) {
                break;
            }
            // 校验有效性
            Conversion conversion = convert(json);
            if (!isValid(conversion)) {
                log.info("[{}] - 无效的转化：{}", ticketConversionDimension, conversion.getRequest().getKeyValueMap());
                continue;
            }
            result.add(json);
        } while (result.size() < size);

        return convert(result);
    }

    private boolean isValid(Conversion conversion) {
        KeyValueMap inadsLog = conversion.getRequest().getKeyValueMap();
        ClickTracker clickTracker = getClickTracker();
        if (Objects.isNull(clickTracker)) {
            log.warn("[{}] - No clickTracker!", getDimension());
            return true;
        }
        String clickId = clickTracker.getClickIdOnInadsAdClickLog(inadsLog);
        return isValid(clickId);
    }

    /**
     * 从点击队列里获取指定 {@code size} 个数据。
     * @param size 数量
     * @return 结果集
     */
    private List<Conversion> getOnClickQueue(int size) {
        List<Conversion> result = Lists.newArrayList();
        List<ClickLog> clickLogs = searchOnClickQueue(size);

        clickLogs.forEach(e -> {
            FeedbackRequest request = of(e);
            result.add(new Conversion().setRequest(request).setClickLog(e));
        });
        return result;
    }

    /**
     * 按照媒体转换数降序，取对应 {@code size} 个点击
     *
     * @param size 数量
     * @return 点击
     */
    private List<ClickLog> searchOnClickQueue(int size) {
        RatioControlConfig config = getConfig();
        String dimension = getDimension();

        int offset = 0;
        int step = 10;
        List<ClickLog> headList = Lists.newArrayList();
        List<ClickLog> helpList = Lists.newArrayList();

        FINISH:
        do {
            // start = 0, stop = 9
            // start = 10, stop = 19
            // ...
            int start = offset * step;
            int stop = (offset + 1) * step - 1;
            Set<String> topN = control.getTopN(start, stop);
            if (CollectionUtils.isEmpty(topN)) {
                // 所有的媒体都已经查找结束了
                if (log.isWarnEnabled()) {
                    log.warn("[{}] 没有策略排行单在点击时 {}-{}", config.getDimension(), start, stop);
                }
                break;
            }
            for (String app : topN) {
                String key = listClickQueue(config.getDuration(), config.getDimension(), app).generateKey();
                String clickLogJson;
                do {
                    clickLogJson = leftPop(key);
                    if (StringUtils.isBlank(clickLogJson)) {
                        // 这个 app 已经没有更多点击了
                        break;
                    }
                    ClickLog clickLog = JsonHelper.convert(clickLogJson, ClickLog.class);
                    if (!isValid(clickLog)) {
                        log.info("[{}] - 无效的点击：{}", dimension, clickLog.getClickId());
                        continue;
                    }
                    if (!clickLog.isHeadTicket()) {
                        // 助力券
                        if (size > helpList.size()) {
                            // 如果助力列表已经超过需要的数量就不用再添加了。
                            helpList.add(clickLog);
                        }
                    } else {
                        // 头部券
                        headList.add(clickLog);
                    }
                } while (headList.size() < size);

                if (headList.size() >= size) {
                    continue FINISH;
                }
                offset ++;
            }
        } while (headList.size() < size);

        // 如果头部券点击已经满足则直接返回
        if (headList.size() >= size) {
            return headList;
        }

        // 将助力券的点击补充到头部券中。
        int remainHead = size - headList.size();
        for (int j = 0; j < remainHead && j < helpList.size(); j++) {
            headList.add(helpList.get(remainHead));
        }
        if (log.isDebugEnabled()) {
            headList.forEach(o -> {
                log.debug("[{}] From click: adid={}, app={}", config.getDimension(), o.getAdid(), o.getApp());
            });
        }
        return headList;
    }

    private boolean isValid(ClickLog clickLog) {
        // 这个点击的入口进入时间必须有效
        String clickId = clickLog.getClickId();
        return isValid(clickId);
    }

    private boolean isValid(String clickId) {
        RatioControlConfig config = getConfig();
        String dimension = getDimension();
        ClickTracker clickTracker = getClickTracker();
        if (Objects.isNull(clickTracker)) {
            log.warn("[{}] - No clickTracker!", dimension);
            return true;
        }
        KeyValueMap trackerLog = clickTracker.getClickTracker(clickId);
        if (Objects.isNull(trackerLog)) {
            log.warn("[{}] - No found: {}", dimension, clickId);
            return true;
        }
        long timestamp = NumberUtils.toLong(trackerLog.getFirst(Inads.Param.TIME), 0);
        if (timestamp == 0) {
            log.warn("[{}] - Illegal time value: {}", dimension, trackerLog);
            return true;
        }
        LocalDateTime localDateTime = LocalDateTimeUtils.of(timestamp);
        if (config.getDuration() == RatioControlConfig.Duration2.HOUR) {
            return LocalTime.now().getHour() == localDateTime.getHour();
        }
        if (config.getDuration() == RatioControlConfig.Duration2.DAY) {
            return LocalDate.now().equals(localDateTime.toLocalDate());
        }
        return true;
    }

    private String rightPopOneOnKeyQueue(String key) {
        String item = rightPop(key);
        if (StringUtils.isBlank(item)) {
            return null;
        }
        String[] largeId = splitLargeId(item);
        long createTime = NumberUtils.toLong(largeId[0]);
        // 如果当前时间 - 创建时间，超过了设置的过期时间，那么认为是无效的。
        Duration duration = Duration.ofHours(getConfig().getExpireInHours());
        if ((System.currentTimeMillis() - createTime) > duration.toMillis()) {
            if (log.isInfoEnabled()) {
                log.info("{} | Created at {}, Expired hour: {}", item, ofTimeMillis(createTime), duration.toHours());
            }
            return rightPopOneOnKeyQueue(key);
        }
        return largeId[1];
    }

    private FeedbackRequest of(ClickLog clickLog) {
        PositionRtb positionRtb = clickLog.getConfig();
        String targetOneRtb = positionRtb.getTargetOneRtb();
        String targetTwoRtb = positionRtb.getTargetTwoRtb();

        String eventType = targetOneRtb;
        int conversionLevel = FeedbackRequest.SHALLOW_CONVERSION_LEVEL;
        if (getConfig().getConversionLevel() == ConversionLevel.DEEP) {
            eventType = targetTwoRtb;
            conversionLevel = FeedbackRequest.DEEP_CONVERSION_LEVEL;
        }

        KeyValueMap clickKeyValueMap = clickLog.getClickKeyValueMap();

        String referrer = clickKeyValueMap.getFirst(Inads.Param.REFER);
        FeedbackRequest request = FeedbackRequest.builder()
                .config(positionRtb)
                .conversionLevel(conversionLevel)
                .conversionType("0")
                .smartConvType(SmartConvType.CLICK_QUEUE)
                .keyValueMap(clickKeyValueMap)
                .eventType(eventType)
                .referrer(referrer)
                .appid(clickLog.getApp())
                .build();
        request.setAdGroupId(fetchAdGroupId(getInstance(), request));
        return request;
    }

    private String fetchAdGroupId(RtbIntegration integration, FeedbackRequest request) {
        String adGroupId = null;
        if (integration instanceof ClickTracker) {
            ClickTracker clickTracker = (ClickTracker) integration;
            adGroupId = clickTracker.getAdGroupId(request);
        }
        return adGroupId;
    }

    private void leftPush(String key, String...members) {
        try (Jedis jedis = getJedisPool().getResource()) {
            jedis.lpush(key, members);
            jedis.expire(key, getConfig().getExpireInSeconds());
        } catch (Exception e) {
            log.error("lpush: {}", e.getMessage());
        }
    }

    private String leftPop(String key) {
        try (Jedis jedis = getJedisPool().getResource()) {
            return jedis.lpop(key);
        } catch (Exception e) {
            log.error("lpop: {}", e.getMessage());
            return null;
        }
    }

    private String rightPop(String key) {
        try (Jedis jedis = getJedisPool().getResource()) {
            return jedis.rpop(key);
        } catch (Exception e) {
            log.error("rpop: {}", e.getMessage());
            return null;
        }
    }

    private ClickTracker getClickTracker() {
        RatioControlBus bus = this.control.getBus();
        RtbIntegration instance = bus.getInstance();
        if (instance instanceof ClickTracker) {
            return (ClickTracker) instance;
        }
        return null;
    }

    @Data
    @Accessors(chain = true)
    private static class Conversion {


        private FeedbackRequest request;
        private ClickLog clickLog;

        public boolean isFromClick() {
            return Objects.nonNull(clickLog);
        }
    }
}