package com.bxm.adscounter.rtb.common.impl;

import com.alibaba.fastjson.JSONException;
import com.bxm.adscounter.rtb.common.FailType;
import com.bxm.adscounter.rtb.common.RtbIntegration;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.utils.HttpClientUtils;
import com.google.common.collect.Lists;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.util.EntityUtils;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Objects;

/**
 * 抽象的 Http 实现。
 *
 * @author allen
 * @date 2022-05-13
 * @since 1.0
 */
public abstract class AbstractHttpRtbIntegration implements RtbIntegration {

    private final HttpClient httpClient;
    protected final RtbConfig config;
    private final static List<Integer> SUCCESS_HTTP_CODE = Lists.newArrayList(HttpStatus.SC_OK);

    protected AbstractHttpRtbIntegration(RtbConfig config) {
        this(config, createHttpClient(config));
    }

    protected AbstractHttpRtbIntegration(RtbConfig config, HttpClient httpClient) {
        this.config = config;
        this.httpClient = httpClient;
    }

    public static HttpClient createHttpClient(RtbConfig config) {
        int maxTotal = config.getMaxTotal();
        int defaultMaxPerRoute = config.getDefaultMaxPerRoute();
        int connectionRequestTimeout = config.getConnectionRequestTimeout();
        int connectTimeout = config.getConnectTimeout();
        int socketTimeout = config.getSocketTimeout();
        return HttpClientUtils.createHttpClient(maxTotal, defaultMaxPerRoute, connectionRequestTimeout, connectTimeout, socketTimeout);
    }

    /**
     * 构造一个HTTP请求对象。
     * @param request 请求对象
     * @return GET / POST 等请求对象。
     */
    protected abstract HttpRequestBase create(FeedbackRequest request) throws RtbIntegrationException;

    /**
     * 返回字符串转化成 FeedbackResponse 对象。
     * @param request 请求对象
     * @param body HTTP请求返回内容
     * @return 返回结果如果为空则按失败处理。
     */
    protected abstract FeedbackResponse convert(FeedbackRequest request, String body) throws RtbIntegrationException;

    /**
     * 是否需要读取响应内容。
     * @return 默认：true
     */
    protected boolean isReadBodyForHttpResponse() {
        return true;
    }

    /**
     * 代表成功响应的状态码
     * @return 默认：true
     */
    protected List<Integer> successHttpCode() {
        return SUCCESS_HTTP_CODE;
    }

    @Override
    public String getFeedbackUrl() {
        return config.getUrl();
    }

    @Override
    public FeedbackResponse doFeedback(FeedbackRequest request) throws RtbIntegrationException {
        return doFeedback(request, 0);
    }

    @Override
    public FeedbackResponse doFeedback(FeedbackRequest request, int status) throws RtbIntegrationException {
        HttpRequestBase requestBase = null;
        try {
            if (Objects.isNull(request)) {
                throw new RtbIntegrationException(FailType.IllegalParameter, String.format("[%s] request object is null", rtb()));
            }
            requestBase = create(request);
            request.setRequestUrl(requestBase.getURI().toString());
            HttpResponse response = httpClient.execute(requestBase);
            int statusCode = response.getStatusLine().getStatusCode();
            if (!successHttpCode().contains(statusCode)) {
                throw new RtbIntegrationException(FailType.ScNotOk, String.format("[%s] response statusCode is %d", rtb(), statusCode));
            }
            boolean readBodyForHttpResponse = isReadBodyForHttpResponse();
            String body = readBodyForHttpResponse ? EntityUtils.toString(response.getEntity(), StandardCharsets.UTF_8) : null;
            FeedbackResponse feedbackResponse = convert(request, body);
            if (Objects.isNull(feedbackResponse)) {
                throw new RtbIntegrationException(FailType.ResponseIsNull, "convert feedback response is null");
            }
            feedbackResponse.setBody(body);
            return feedbackResponse;
        } catch (IOException e) {
            throw new RtbIntegrationException(FailType.IoException, e);
        } catch (JSONException e) {
            throw new RtbIntegrationException(FailType.ResolveProtocolException, e);
        } catch (RtbIntegrationException e) {
            throw e;
        } catch (Exception e) {
            throw new RtbIntegrationException(FailType.OtherException, e);
        } finally {
            if (Objects.nonNull(requestBase)) {
                requestBase.releaseConnection();
            }
        }
    }
}
