package com.bxm.adscounter.rtb.common.impl.baidu;

import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.impl.AbstractClickTrackerRtbIntegration;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.UrlHelper;
import com.google.common.collect.Lists;
import lombok.Data;
import lombok.experimental.Accessors;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpHeaders;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.StringEntity;
import org.springframework.http.MediaType;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import java.nio.charset.StandardCharsets;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * @author allen
 * @date 2022-05-16
 * @since 1.0
 */
public class BaiduRtbIntegration extends AbstractClickTrackerRtbIntegration {

    public static final String CLICK_ID = "bd_vid";
    public static final String AD_GROUP_ID = "ad_group_id";

    public BaiduRtbIntegration(BaiduConfig config) {
        super(config);
    }

    @Override
    public String getClickId(KeyValueMap clickTrackerKeyValueMap) {
        return clickTrackerKeyValueMap.getFirst(Common.Param.CLICK_ID);
    }

    @Override
    public String getClickIdOnInadsAdClickLog(KeyValueMap clickEventLog) {
        return UrlHelper.getFirstValueOfParamName(clickEventLog.getRef(), CLICK_ID);
    }

    @Override
    public String getAdGroupId(FeedbackRequest request) {
        String referrer = request.getReferrer();
        UriComponents build = UriComponentsBuilder.fromUriString(referrer).build();
        MultiValueMap<String, String> queryParams = build.getQueryParams();
        String clickId = queryParams.getFirst(CLICK_ID);
        KeyValueMap clickTracker = getClickTracker(clickId);

        return Optional.ofNullable(clickTracker)
                .map(k -> k.getFirst(Common.Param.AD_GROUP_ID))
                .orElse(StringUtils.EMPTY);
    }

    @Override
    protected HttpRequestBase create(FeedbackRequest request) throws RtbIntegrationException {
        HttpPost post = new HttpPost(config.getUrl());
        post.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_UTF8_VALUE);
        BaiduRequest req = new BaiduRequest().setToken(request.getConfig().getToken());
        Long time = LocalDateTime.now().toEpochSecond(ZoneOffset.of("+8"));
        ConversionType conversionType = new ConversionType()
                .setLogidUrl(request.getReferrer())
                .setNewType(request.getEventType())
                .setConvertTime(time);
        req.setConversionTypes(Lists.newArrayList(conversionType));
        String entity = JsonHelper.convert(req);
        post.setEntity(new StringEntity(entity, StandardCharsets.UTF_8));
        request.setRequestBody(entity);
        return post;
    }

    @Override
    protected FeedbackResponse convert(FeedbackRequest request, String body) throws RtbIntegrationException {
        return JsonHelper.convert(body, BaiduResponse.class);
    }

    @Override
    public Rtb rtb() {
        return Rtb.Baidu;
    }

    @Override
    public String[] getMissingAdGroupIdList() {
        return new String[] {"__UNIT_ID__", "NULL"};
    }

    @Data
    @Accessors(chain = true)
    private static class BaiduRequest {

        private String token;
        private List<ConversionType> conversionTypes;
    }

    @Data
    @Accessors(chain = true)
    private static class ConversionType {

        private String logidUrl;
        private String newType;
        private Long convertTime;
    }

    private static class BaiduResponse extends FeedbackResponse {

        private Header header;

        @Override
        public boolean isSuccess() {
            return Objects.nonNull(header) && header.isSuccess();
        }

        public Header getHeader() {
            return header;
        }

        public void setHeader(Header header) {
            this.header = header;
        }
    }

    private static class Header {

        private String desc;
        private String status;

        public boolean isSuccess() {
            return StringUtils.equals("0", status);
        }

        public String getDesc() {
            return desc;
        }

        public void setDesc(String desc) {
            this.desc = desc;
        }

        public String getStatus() {
            return status;
        }

        public void setStatus(String status) {
            this.status = status;
        }
    }
}
