package com.bxm.adscounter.rtb.common.impl.bilibili;

import com.bxm.adscounter.rtb.common.FailType;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.impl.AbstractClickTrackerRtbIntegration;
import com.bxm.adscounter.rtb.common.impl.AbstractHttpRtbIntegration;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.UrlHelper;
import com.bxm.warcar.utils.http.OkHttpUtils;
import com.google.common.collect.Maps;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;

/**
 * @author tangxiao
 * @date 2022-08-25
 * @since 1.0
 */
@Slf4j
public class BilibiliRtbIntegration extends AbstractClickTrackerRtbIntegration {

    public static final String CLICK_ID = "trackid";
    public static final String AD_GROUP_ID = "ad_group_id";

    public BilibiliRtbIntegration(BilibiliConfig config) {
        super(config);
    }

    @Override
    public Rtb rtb() {
        return Rtb.Bilibili;
    }

    @Override
    public String getClickId(KeyValueMap clickTrackerKeyValueMap) {
        return clickTrackerKeyValueMap.getFirst(Common.Param.CLICK_ID);
    }

    @Override
    public String getClickIdOnInadsAdClickLog(KeyValueMap clickEventLog) {
        return UrlHelper.getFirstValueOfParamName(clickEventLog.getRef(), CLICK_ID);
    }

    @Override
    public String getAdGroupId(FeedbackRequest request) {
        String referrer = request.getReferrer();
        UriComponents build = UriComponentsBuilder.fromUriString(referrer).build();
        MultiValueMap<String, String> queryParams = build.getQueryParams();
        String clickId = queryParams.getFirst(CLICK_ID);
        KeyValueMap clickTracker = getClickTracker(clickId);

        return Optional.ofNullable(clickTracker)
                .map(k -> k.getFirst(Common.Param.AD_GROUP_ID))
                .orElse(StringUtils.EMPTY);
    }

    @Override
    protected boolean isReadBodyForHttpResponse() {
        return true;
    }

    @Override
    protected HttpRequestBase create(FeedbackRequest request) throws RtbIntegrationException {
        String referrer = request.getReferrer();
        String eventType = request.getEventType();

        MultiValueMap<String, String> queryParams = UriComponentsBuilder.fromUriString(referrer).build().getQueryParams();
        String trackId = queryParams.getFirst("trackid");
        if (StringUtils.isBlank(trackId)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'trackid' by referrer.");
        }

        // trackId解码两次
        String trackIdDecode = UrlHelper.urlDecode(UrlHelper.urlDecode(trackId));

        Map<String, Object> params = Maps.newHashMap();
        params.put("track_id", trackIdDecode);
        params.put("conv_type", eventType);
        params.put("conv_time", System.currentTimeMillis());

        String feedbackUrl = getFeedbackUrl();
        feedbackUrl = OkHttpUtils.appendParams(feedbackUrl, params);
        return new HttpGet(feedbackUrl);
    }

    @Override
    protected FeedbackResponse convert(FeedbackRequest request, String body) {
        BilibiliResponse response = JsonHelper.convert(body, BilibiliResponse.class);
        FeedbackResponse feedbackResponse = new FeedbackResponse();
        if (Objects.nonNull(response)) {
            feedbackResponse.setSuccess(response.isSuccess());
            feedbackResponse.setBody(response.getMessage());
        } else {
            feedbackResponse.setSuccess(false);
        }
        return feedbackResponse;
    }

    @Data
    private static class BilibiliResponse {
        private String code;
        private String message;
        private String data;

        public boolean isSuccess() {
            return StringUtils.equalsIgnoreCase("0", code);
        }
    }

}
