package com.bxm.adscounter.rtb.common.impl.iqiyi;

import com.bxm.adscounter.rtb.common.FailType;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.impl.AbstractClickTrackerRtbIntegration;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.openlog.sdk.consts.Inads;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.UrlHelper;
import com.bxm.warcar.utils.http.OkHttpUtils;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import java.math.BigDecimal;
import java.util.Map;
import java.util.Optional;

/**
 * @author allen
 * @date 2022-05-16
 * @since 1.0
 */
@Slf4j
public class IqiyiRtbIntegration extends AbstractClickTrackerRtbIntegration {

    public static final String CLICK_ID = "impress_id";

    private static final String EVENT_TYPE_APP_ORDER = "206";

    public IqiyiRtbIntegration(IqiyiConfig config) {
        super(config);
    }

    @Override
    public String getClickId(KeyValueMap clickTrackerKeyValueMap) {
        return clickTrackerKeyValueMap.getFirst(Common.Param.CLICK_ID);
    }

    @Override
    public String getClickIdOnInadsAdClickLog(KeyValueMap clickEventLog) {
        return UrlHelper.getFirstValueOfParamName(clickEventLog.getRef(), CLICK_ID);
    }

    @Override
    public String getAdGroupId(FeedbackRequest request) {
        String referrer = request.getReferrer();
        UriComponents build = UriComponentsBuilder.fromUriString(referrer).build();
        MultiValueMap<String, String> queryParams = build.getQueryParams();
        String clickId = queryParams.getFirst(CLICK_ID);
        KeyValueMap clickTracker = getClickTracker(clickId);

        return Optional.ofNullable(clickTracker)
                .map(k -> k.getFirst(Common.Param.AD_GROUP_ID))
                .orElse(StringUtils.EMPTY);
    }

    @Override
    public Rtb rtb() {
        return Rtb.iQiyi;
    }

    @Override
    protected HttpRequestBase create(FeedbackRequest request) throws RtbIntegrationException {
        String url = getFeedbackUrl();
        String referrer = request.getReferrer();
        String eventType = request.getEventType();

        String impressId = UriComponentsBuilder.fromUriString(referrer).build().getQueryParams().getFirst(CLICK_ID);
        if (StringUtils.isBlank(impressId)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'impressId' by referrer.");
        }

        Map<String, Object> params = Maps.newHashMap();
        params.put("conv_time", String.valueOf(System.currentTimeMillis() / 1000));
        params.put("event_type", eventType);
        params.put("impress_id", impressId);

        if (EVENT_TYPE_APP_ORDER.equals(eventType)) {
            String payAmount = request.getKeyValueMap().getFirst(Inads.Param.PAY_AMOUNT);
            if (StringUtils.isNotBlank(payAmount)) {
                // 元转分
                int payAmountCent = new BigDecimal(payAmount).multiply(new BigDecimal(100)).intValue();
                params.put("pmao", payAmountCent);
            }
        }

        url = OkHttpUtils.appendParams(url, params);
        return new HttpGet(url);
    }

    @Override
    protected FeedbackResponse convert(FeedbackRequest request, String body) throws RtbIntegrationException {
        return JsonHelper.convert(body, Response.class);
    }

    private static class Response extends FeedbackResponse {

        private String status;
        private String message;

        @Override
        public boolean isSuccess() {
            return StringUtils.equals("200", status);
        }

        public String getStatus() {
            return status;
        }

        public void setStatus(String status) {
            this.status = status;
        }

        public String getMessage() {
            return message;
        }

        public void setMessage(String message) {
            this.message = message;
        }
    }
}
