package com.bxm.adscounter.rtb.common.impl.juliang;

import com.alibaba.fastjson.JSONObject;
import com.bxm.adscounter.model.constant.Constants;
import com.bxm.adscounter.rtb.common.FailType;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.impl.AbstractClickTrackerRtbIntegration;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.UrlHelper;
import lombok.Data;
import lombok.experimental.Accessors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpHeaders;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.StringEntity;
import org.springframework.http.MediaType;
import org.springframework.web.util.UriComponentsBuilder;

import java.nio.charset.StandardCharsets;
import java.util.Optional;

/**
 * @author daill
 * @date 2022-05-16
 * @since 1.0
 */
@Slf4j
public class OceanEngineRtbIntegration extends AbstractClickTrackerRtbIntegration {

    public static final String CLICK_ID = "clickid";
    public static final String AD_GROUP_ID = "adid";

    public OceanEngineRtbIntegration(OceanEngineConfig config) {
        super(config);
    }

    @Override
    public String getClickId(KeyValueMap clickTrackerKeyValueMap) {
        return clickTrackerKeyValueMap.getFirst(Common.Param.CLICK_ID);
    }

    @Override
    public String getClickIdOnInadsAdClickLog(KeyValueMap clickEventLog) {
        return UrlHelper.getFirstValueOfParamName(clickEventLog.getRef(), CLICK_ID);
    }

    @Override
    public String getAdGroupId(FeedbackRequest request) {
        String clickId = getClickIdOnInadsAdClickLog(request.getKeyValueMap());
        KeyValueMap clickTracker = getClickTracker(clickId);
        return Optional.ofNullable(clickTracker)
                .map(k -> k.getFirst(Common.Param.AD_GROUP_ID))
                .orElse(StringUtils.EMPTY);
    }

    @Override
    protected HttpRequestBase create(FeedbackRequest request) throws RtbIntegrationException {
        String url = getFeedbackUrl();
        String referrer = request.getReferrer();
        String eventType = request.getEventType();
        KeyValueMap keyValueMap = request.getKeyValueMap();

        String callBack = request.getClickId();
        if (StringUtils.isBlank(callBack)) {
            callBack = UriComponentsBuilder.fromUriString(referrer).build().getQueryParams().getFirst(CLICK_ID);
        }
        if (StringUtils.isBlank(callBack)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'clickid' by referrer.");
        }
        AD ad = new AD().setCallback(callBack);
        Context context = new Context().setAd(ad);
        //主要 巨量引擎的 时间戳只支持字符串
        OceanEngineRequest oceanEngineRequest = new OceanEngineRequest()
                .setEvent_type(eventType)
                .setTimestamp(System.currentTimeMillis())
                .setContext(context);

        fillMeituanOcpsParam(oceanEngineRequest, keyValueMap);

        String entity = JsonHelper.convert(oceanEngineRequest);
        HttpPost post = new HttpPost(url);
        post.setEntity(new StringEntity(entity, StandardCharsets.UTF_8));
        post.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_UTF8_VALUE);
        request.setRequestBody(entity);
        return post;
    }

    private void fillMeituanOcpsParam(OceanEngineRequest oceanEngineRequest, KeyValueMap keyValueMap) {
        Properties properties = new Properties();
        if (keyValueMap.containsKey("order_cnt")) {
            properties.setOrder_cnt(keyValueMap.getFirst(Constants.MeituanOcps.DIRECT));
        }
        if (keyValueMap.containsKey("pay_amount")) {
            properties.setPay_amount(keyValueMap.getFirst(Constants.MeituanOcps.QUANTITY));
        }
        if (properties.isNotEmpty()) {
            log.info("OceanEngine meituan order_cnt: {}", JSONObject.toJSONString(properties));
            oceanEngineRequest.setProperties(properties);
        }
    }

    @Override
    public Rtb rtb() {
        return Rtb.JuLiangYinQing;
    }

    @Override
    protected FeedbackResponse convert(FeedbackRequest request, String body) {
        return JsonHelper.convert(body, Response.class);
    }

    private static class Response extends FeedbackResponse {

        private String code;
        private String message;

        @Override
        public boolean isSuccess() {
            return StringUtils.equals("0", code);
        }

        public String getCode() {
            return code;
        }

        public void setCode(String code) {
            this.code = code;
        }

        public String getMessage() {
            return message;
        }

        public void setMessage(String message) {
            this.message = message;
        }
    }

    @Data
    @Accessors(chain = true)
    private static class AD {
        private String callback;
    }
    @Data
    @Accessors(chain = true)
    private static class Context {
        private AD ad;
    }
    @Data
    @Accessors(chain = true)
    private static class OceanEngineRequest {

        private String event_type;
        private Context context;
        private Properties properties;
        private Long timestamp;
    }

    @Data
    @Accessors(chain = true)
    private static class Properties {
        private String pay_amount;
        private String order_cnt;

        public boolean isNotEmpty() {
            return StringUtils.isNotBlank(pay_amount) || StringUtils.isNotBlank(order_cnt);
        }
    }
}
