package com.bxm.adscounter.rtb.common.impl.kuaishou;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.annotation.JSONField;
import com.bxm.adscounter.rtb.common.ClickTracker;
import com.bxm.adscounter.rtb.common.FailType;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.impl.AbstractClickTrackerRtbIntegration;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.Production;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.openlog.sdk.consts.Inads;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.UrlHelper;
import com.bxm.warcar.utils.http.OkHttpUtils;
import com.google.common.collect.Maps;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;

/**
 * @author allen
 * @date 2022-05-13
 * @since 1.0
 */
@Slf4j
public class KuaishouRtbIntegration extends AbstractClickTrackerRtbIntegration implements ClickTracker {

    public static final String CLICK_ID = "callback";
    private final KuaishouConfig config;

    private static final String EVENT_TYPE_FORM = "9";

    public KuaishouRtbIntegration(KuaishouConfig config) {
        super(config);
        this.config = config;
    }

    @Override
    public boolean forceSaveClickTracker() {
        return true;
    }

    @Override
    public String getClickId(KeyValueMap clickTrackerKeyValueMap) {
        // 快手的点击监测中参数 callback 值里的 callback 才是进入落地页时的 callback。
        String callback = clickTrackerKeyValueMap.getFirst(Common.Param.CLICK_ID);
        if (StringUtils.isBlank(callback)) {
            log.info("Empty value for callback!");
            return null;
        }
        if (StringUtils.equals(clickTrackerKeyValueMap.getProduction(), Production.INADS.getName())
                && StringUtils.equals(clickTrackerKeyValueMap.getMt(), Inads.Mt.IndexShow.original())) {
            return callback;
        }
        String url = UrlHelper.urlDecode(callback);
        return UrlHelper.getFirstValueOfParamName(url, CLICK_ID);
    }

    @Override
    public String getClickIdOnInadsAdClickLog(KeyValueMap clickEventLog) {
        String ref = clickEventLog.getRef();
        return UrlHelper.getFirstValueOfParamName(ref, CLICK_ID);
    }

    @Override
    public String getAdGroupId(FeedbackRequest request) {
        String referrer = request.getReferrer();
        UriComponents build = UriComponentsBuilder.fromUriString(referrer).build();
        MultiValueMap<String, String> queryParams = build.getQueryParams();
        String clickId = request.getClickId();
        if (StringUtils.isBlank(clickId)) {
            clickId = queryParams.getFirst(CLICK_ID);
        }

        KeyValueMap clickTracker = getClickTracker(clickId);

        return Optional.ofNullable(clickTracker)
                .map(k -> k.getFirst(Common.Param.AD_GROUP_ID))
                .orElse(StringUtils.EMPTY);
    }

    @Override
    public String getApp(KeyValueMap keyValueMap) {
        String ext = keyValueMap.getFirst(Inads.Param.EXT);
        if (StringUtils.isBlank(ext)) {
            return null;
        }
        String app = null;
        try {
            JSONObject jsonObject = JSONObject.parseObject(ext);
            app = jsonObject.getString("x_pkgname");
        } catch (Exception e) {
            if (log.isDebugEnabled()) {
                log.debug("parseObject: '{}' on log: {}", e.getMessage(), keyValueMap);
            }
        }
        return app;
    }

    @Override
    protected HttpRequestBase create(FeedbackRequest request) throws RtbIntegrationException {
        String eventType = request.getEventType();
        String referrer = request.getReferrer();
        String extend = getExtendAction(request);

        // 先从clickId中获取
        String callback = request.getClickId();
        if (StringUtils.isBlank(callback)) {
            UriComponents build = UriComponentsBuilder.fromUriString(referrer).build();
            MultiValueMap<String, String> queryParams = build.getQueryParams();
            callback = queryParams.getFirst(CLICK_ID);
        }
        if (StringUtils.isBlank(callback)) {
            String bxmId = getBxmId(request);
            throw new RtbIntegrationException(FailType.NotFoundClickId, "[" + bxmId + "] 无法找到 callback 值，在 Referrer (" + referrer + ") 中。");
        }

        Map<String, Object> params = Maps.newHashMap();
        params.put("callback", callback);
        params.put("event_type", eventType);
        params.put("event_time", String.valueOf(System.currentTimeMillis()));

        boolean isForm = Objects.equals(EVENT_TYPE_FORM, eventType);
        if (isForm && StringUtils.isNotBlank(extend)) {
            params.put("event_props", buildEventProps(extend));
        }

        String finalUrl = OkHttpUtils.appendParams(config.getUrl(), params);

        return new HttpGet(finalUrl);
    }

    private String getExtendAction(FeedbackRequest request) {
        PositionRtb positionRtb = request.getConfig();
        //ocpx没有这个配置，排除空指针异常
        if (positionRtb == null) {
            return null;
        }
        String extend = positionRtb.getExtend();
        if (StringUtils.isNotBlank(extend)) {
            ActionExtend actionExtend = JsonHelper.convert(extend, ActionExtend.class);
            extend = request.isDeepConversion() ? actionExtend.getDeepAction() : actionExtend.getShallowAction();
        }
        return extend;
    }

    private String getBxmId(FeedbackRequest request) {
        KeyValueMap keyValueMap = request.getKeyValueMap();
        return Optional.ofNullable(keyValueMap)
                .map(e -> e.getFirst(Inads.Param.BXMID))
                .orElse("unknown");
    }

    private String buildEventProps(String extend) {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("actionName", "EXTERNAL_ACTION" + extend);
        return jsonObject.toJSONString();
    }

    @Override
    protected FeedbackResponse convert(FeedbackRequest request, String body) throws RtbIntegrationException {
        return JsonHelper.convert(body, KuaishouResponse.class);
    }

    @Override
    public Rtb rtb() {
        return Rtb.Kuaishou;
    }

    private static class KuaishouResponse extends FeedbackResponse {

        private String result;
        @JSONField(name = "host-name")
        private String hostName;

        @Override
        public boolean isSuccess() {
            return StringUtils.equalsIgnoreCase("1", result);
        }

        public String getResult() {
            return result;
        }

        public void setResult(String result) {
            this.result = result;
        }

        public String getHostName() {
            return hostName;
        }

        public void setHostName(String hostName) {
            this.hostName = hostName;
        }
    }

    @Data
    public static class ActionExtend {
        private String shallowAction;
        private String deepAction;
    }
}
