package com.bxm.adscounter.rtb.common.impl.oppo;

import com.bxm.adscounter.rtb.common.FailType;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.impl.AbstractClickTrackerRtbIntegration;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.openlog.sdk.consts.Inads;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.UrlHelper;
import lombok.Data;
import lombok.experimental.Accessors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpHeaders;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.StringEntity;
import org.springframework.http.MediaType;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import java.nio.charset.StandardCharsets;

/**
 * @author daill
 * @date 2022-06-30
 * @since 1.0
 */
@Slf4j
public class OppoRtbIntegration extends AbstractClickTrackerRtbIntegration {

    public static final String CLICK_ID = "tid";
    private final OppoConfig config;

    public OppoRtbIntegration(OppoConfig config) {
        super(config);
        this.config = config;
    }

    @Override
    public String getClickId(KeyValueMap clickTrackerKeyValueMap) {
        return clickTrackerKeyValueMap.getFirst(Common.Param.CLICK_ID);
    }

    @Override
    public String getClickIdOnInadsAdClickLog(KeyValueMap clickEventLog) {
        return UrlHelper.getFirstValueOfParamName(clickEventLog.getRef(), CLICK_ID);
    }

    @Override
    public String getAdGroupId(FeedbackRequest request) {
        String referrer = request.getReferrer();
        // https://alidocs.dingtalk.com/i/team/BpLmkK6448BpPm2g/docs/BpLmkwY47dKOam2g# 「OPPO支持cvr控制、扣量回传、控量回传」

        // https://m.cat-happy.cn/dist/welfareAT02/private/T/T148/index17.html?business=money-1
        // &appkey=48e74bc775d3463bac76ecbb302d4cc9&uid=B6B7A8B8EF283287EDFFB08791BE75FE
        // &fmmodel_id=actNotColdStart,actSendTicketCount1,actCvr1&activityid=24683&spm=20220818165345487000016611314521
        // &tid=1-kkjMD6zV8KXhcPEViRkz1Z0oTSb9f8RqIdoJFKHy0hGU0wzQjgWjTvdiMuc6R1o9Yqdf5Jnn0yFHRDxAnoTiN8xeLCPXIbXVq3+Z/vWXYyiOaQ5gYL0VebGuk/m5Gjmh
        // &req_id=acd78227-57bb-4aa6-95a3-3d436a7765d9-1660722920125-6&lbid=1_0_6-01&pageId=1200020197
        // &transparent=413848717_1000170732_202388492_304367154#

        // 可以从 ClickTracker 里取，因为 OppoIndexShowEventListener 保存了它们的关系，但是 Referrer 中已经有了就少查询一次了。

        UriComponents build = UriComponentsBuilder.fromUriString(referrer).build();
        MultiValueMap<String, String> queryParams = build.getQueryParams();
        return getAdGroupIdForTransparent(queryParams.getFirst("transparent"));
    }

    public String getAdGroupIdForTransparent(String transparent) {
        if (StringUtils.isBlank(transparent)) {
            return null;
        }
        String[] arr = StringUtils.split(transparent, "_");
        return arr[arr.length - 1];
    }

    @Override
    protected HttpRequestBase create(FeedbackRequest request) throws RtbIntegrationException {
        String url = getFeedbackUrl();
        String referrer = request.getReferrer();
        String eventType = request.getEventType();
        PositionRtb positionRtb = request.getConfig();
        String pageId = UriComponentsBuilder.fromUriString(referrer).build().getQueryParams().getFirst("pageId");
        if (StringUtils.isBlank(pageId)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'pageId' by referrer.");
        }
        String tid = UriComponentsBuilder.fromUriString(referrer).build().getQueryParams().getFirst("tid");
        if (StringUtils.isBlank(tid)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'tid' by referrer.");
        }
        String lbid = UriComponentsBuilder.fromUriString(referrer).build().getQueryParams().getFirst("lbid");
        if (StringUtils.isBlank(lbid)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'lbid' by referrer.");
        }
        String ip = request.getKeyValueMap().getFirst(Inads.Param.IP);
        OPPORequest oppoRequest = new OPPORequest()
                .setPageId(pageId)
                .setOwnerId(positionRtb.getCustomerId())
                .setIp(ip)
                .setTid(UrlHelper.urlEncode(tid))
                .setLbid(lbid)
                .setTransformType(eventType);

        String entity = JsonHelper.convert(oppoRequest);
        HttpPost post = new HttpPost(url);
        post.addHeader(HttpHeaders.AUTHORIZATION, "Bearer " + getToken(positionRtb.getSourceId(), positionRtb.getToken(), positionRtb.getCustomerId()));
        post.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_UTF8_VALUE);
        post.setEntity(new StringEntity(entity, StandardCharsets.UTF_8));
        return post;
    }

    private static String getToken(String apiId, String apiKey, String ownerId) {
        try {
            // sign=sha1(api_id+api_key+time_stamp)，
            long timestamp = System.currentTimeMillis() / 1000;
            String sign = DigestUtils.sha1Hex(apiId + apiKey + timestamp);
            // token=base64(owner_id+“,”+api_id+“,”+time_stamp+“,”+sign)
            String str = ownerId + "," + apiId + "," + timestamp + "," + sign;
            return Base64.encodeBase64String(str.getBytes());
        } catch (Exception e) {
            return null;
        }
    }

    @Override
    public Rtb rtb() {
        return Rtb.OPPO;
    }

    @Override
    protected FeedbackResponse convert(FeedbackRequest request, String body) {
        return JsonHelper.convert(body, Response.class);
    }

    @Override
    public String[] getMissingAdGroupIdList() {
        return new String[] {"NULL"};
    }

    private static class Response extends FeedbackResponse {

        private String code;
        private String message;
        private String data;

        @Override
        public boolean isSuccess() {
            return StringUtils.equals("0", code);
        }

        public String getCode() {
            return code;
        }

        public void setCode(String code) {
            this.code = code;
        }

        public String getMessage() {
            return message;
        }

        public void setMessage(String message) {
            this.message = message;
        }

        public String getData() {
            return data;
        }

        public void setData(String data) {
            this.data = data;
        }
    }

    @Data
    @Accessors(chain = true)
    private static class OPPORequest {
        private String pageId;
        private String ownerId;
        private String ip;
        private String tid;
        private String lbid;
        private String transformType;
    }
}
