package com.bxm.adscounter.rtb.common.impl.qimao;

import com.alibaba.fastjson.JSONObject;
import com.bxm.adscounter.integration.utils.HttpClientUtils;
import com.bxm.adscounter.model.constant.RedisKeyGenerator;
import com.bxm.adscounter.rtb.common.FailType;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.impl.AbstractClickTrackerRtbIntegration;
import com.bxm.adscounter.rtb.common.impl.RtbConfig;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.UrlHelper;
import com.google.common.collect.Sets;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.ssl.SSLContexts;
import org.apache.logging.log4j.util.Strings;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.web.util.UriComponentsBuilder;

import javax.net.ssl.SSLContext;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.Optional;
import java.util.Set;

/**
 * 七猫使用设备号作为归因参数，用于获取点击监测参数
 * click_id在点击监测中
 * @author tangxiao
 * @date 2023-06-29
 * @since 1.0
 */
@Slf4j
public class QiMaoRtbIntegration extends AbstractClickTrackerRtbIntegration {

    private final Fetcher fetcher;

    /**
     * 用于点击监测中获取设备号
     */
    private static final String[] clickTrackerDeviceIdTypes = {
            "imeiMd5",
            "idfa",
            "idfaMd5",
            "anid",
            "anidMd5",
            "oaid",
            "oaidMd5"
    };

    /**
     * 用于互动中获取设备号
     */
    private static final String[] inadsDeviceIdTypes = {
            Common.RtbParam.IMEI_MD5,
            Common.RtbParam.IDFA,
            Common.RtbParam.IDFA_MD5,
            Common.RtbParam.ANDROIDID,
            Common.RtbParam.ANDROIDID_MD5,
            Common.RtbParam.OAID,
            Common.RtbParam.OAID_MD5
    };

    private static final Set<String> deviceIdMacroSet = Sets.newHashSet(
            "__IMEI_MD5__",
            "__IDFA__",
            "__IDFA_MD5__",
            "__ANDROIDID__",
            "__ANDROIDID__MD5__",
            "__OAID__",
            "__OAID__MD5__"
    );

    public QiMaoRtbIntegration(QiMaoConfig config, @Qualifier("rtbJedisFetcher") Fetcher fetcher) {
        super(config, createHttpClient(config));
        this.fetcher = fetcher;
    }

    public static HttpClient createHttpClient(RtbConfig config) {
        SSLContext sslContext = null;
        try {
            sslContext = SSLContexts.custom().setProtocol("TLSv1.2").build();
        } catch (Exception e) {
            log.error("error", e);
        }
        return HttpClientUtils.createHttpClient(config.getMaxTotal(), config.getDefaultMaxPerRoute(), config.getConnectionRequestTimeout(), config.getConnectTimeout(), config.getSocketTimeout(), sslContext);
   }

    @Override
    public String getClickId(KeyValueMap clickTrackerKeyValueMap) {
        return clickTrackerKeyValueMap.getFirst(Common.Param.CLICK_ID);
    }

    @Override
    public String getClickIdOnInadsAdClickLog(KeyValueMap clickEventLog) {
        String deviceId = findDeviceIdInMap(clickEventLog, false);
        if (StringUtils.isNotBlank(deviceId)) {
            return fetcher.fetch(RedisKeyGenerator.strQiMaoDeviceClickId(deviceId), String.class);
        }
        return Strings.EMPTY;
    }

    @Override
    public String getAdGroupId(FeedbackRequest request) {
        String clickId = getClickIdOnInadsAdClickLog(request.getKeyValueMap());
        if (StringUtils.isNotBlank(clickId)) {
            KeyValueMap clickTracker = getClickTracker(clickId);
            return Optional.ofNullable(clickTracker)
                    .map(k -> k.getFirst(Common.Param.AD_GROUP_ID))
                    .orElse(StringUtils.EMPTY);
        }
        return Strings.EMPTY;
    }

    @Override
    public boolean forceSaveClickTracker() {
        return true;
    }

    @Override
    protected HttpRequestBase create(FeedbackRequest request) throws RtbIntegrationException {
        String eventType = request.getEventType();
        KeyValueMap keyValueMap = request.getKeyValueMap();

        String clickId = getClickIdOnInadsAdClickLog(keyValueMap);

        if (StringUtils.isBlank(clickId)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, String.format("Cannot found clickId. keyValueMap: %s", JSONObject.toJSONString(keyValueMap)));
        }

        // 从点击监测里获取callback
        KeyValueMap clickTracker = getClickTracker(clickId);

        if (null == clickTracker) {
            throw new RtbIntegrationException(FailType.IllegalParameter, String.format("Cannot found clickTracker by clickId. clickId: %s", clickId));
        }

        String callback = clickTracker.getFirst(Common.Param.CLICK_ID);

        if (StringUtils.isBlank(callback)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, String.format("Cannot found click_id from clickTracker. clickTracker: %s", JSONObject.toJSONString(clickTracker)));
        }

        String callbackUrl = UriComponentsBuilder.fromUriString(UrlHelper.urlDecode(callback))
                .replaceQueryParam("event_type", eventType)
                .replaceQueryParam("timestamp", System.currentTimeMillis())
                .build()
                .toString();

        return new HttpGet(callbackUrl);
    }

    public static String findDeviceIdInMap(KeyValueMap keyValueMap, boolean isClickTracker) {
        String[] deviceIdTypes = isClickTracker ? clickTrackerDeviceIdTypes : inadsDeviceIdTypes;
        for (String deviceIdType : deviceIdTypes) {
            String deviceId = keyValueMap.getFirst(deviceIdType);
            if (StringUtils.isNotBlank(deviceId) && !deviceIdMacroSet.contains(deviceId)) {
                return deviceId;
            }
        }
        return Strings.EMPTY;
    }

    @Override
    public Rtb rtb() {
        return Rtb.QiMao;
    }

    @Override
    protected FeedbackResponse convert(FeedbackRequest request, String body) {
        return JsonHelper.convert(body, QiMaoResponse.class);

    }

    private static class QiMaoResponse extends FeedbackResponse {

        private int code = -1;
        private String message;

        @Override
        public boolean isSuccess() {
            return code == 0;
        }

        public int getCode() {
            return code;
        }

        public void setCode(int code) {
            this.code = code;
        }

        public String getMessage() {
            return message;
        }

        public void setMessage(String message) {
            this.message = message;
        }
    }

}
