package com.bxm.adscounter.rtb.common.impl.uc;

import com.bxm.adscounter.rtb.common.FailType;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.impl.AbstractClickTrackerRtbIntegration;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.openlog.sdk.consts.Inads;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.UrlHelper;
import com.bxm.warcar.utils.http.OkHttpUtils;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.Map;
import java.util.Optional;

/**
 * @author daill
 * @date 2022-05-16
 * @since 1.0
 */
@Slf4j
public class UcRtbIntegration extends AbstractClickTrackerRtbIntegration {

    public static final String CLICK_ID = "uctrackid";
    public static final String AD_GROUP_ID = "campaignid";

    private final UcConfig config;

    public UcRtbIntegration(UcConfig config) {
        super(config);
        this.config = config;
    }

    @Override
    public String getClickId(KeyValueMap clickTrackerKeyValueMap) {
        return clickTrackerKeyValueMap.getFirst(Common.Param.CLICK_ID);
    }

    @Override
    public String getClickIdOnInadsAdClickLog(KeyValueMap clickEventLog) {
        return UrlHelper.getFirstValueOfParamName(clickEventLog.getRef(), CLICK_ID);
    }

    @Override
    public String getAdGroupId(FeedbackRequest request) {
        String clickId = getClickIdOnInadsAdClickLog(request.getKeyValueMap());
        KeyValueMap clickTracker = getClickTracker(clickId);
        return Optional.ofNullable(clickTracker)
                .map(k -> k.getFirst(Common.Param.AD_GROUP_ID))
                .orElse(StringUtils.EMPTY);
    }

    @Override
    protected HttpRequestBase create(FeedbackRequest request) throws RtbIntegrationException {
        String url = getFeedbackUrl();
        String referrer = request.getReferrer();
        String eventType = request.getEventType();
        String positionId = request.getKeyValueMap().getFirst(Inads.Param.TAGID);
        if (StringUtils.isBlank(positionId)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'tagid'.");
        }

        String uctrackid = UriComponentsBuilder.fromUriString(referrer).build().getQueryParams().getFirst("uctrackid");
        if (StringUtils.isBlank(uctrackid)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'uctrackid' by referrer.");
        }

        String[] arr = StringUtils.split(positionId, "-");
        if (arr.length < 2) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Illegal value of 'tagid'.");
        }
        String appKey = arr[0];
        String entrance = arr[1];
        String linkConfig = request.getConfig().getCustomerId();
        String link ;
        //采用配置的域名
        if (StringUtils.isNotBlank(linkConfig)) {
            link = config.getReturnLinkUrl().replace("i.bianxianmao.com", linkConfig).replace("__APPKEY__", appKey).replace("__APPENNTRANCE__", entrance)
                    + "&uctrackid=" + uctrackid;
        } else {
            link = config.getReturnLinkUrl().replace("__APPKEY__", appKey).replace("__APPENNTRANCE__", entrance)
                    + "&uctrackid=" + uctrackid;
        }


        Map<String, Object> params = Maps.newHashMap();
        params.put("link", link);
        params.put("event_type", eventType);
        params.put("event_time", String.valueOf(System.currentTimeMillis()));
        url = OkHttpUtils.appendParams(url, params);
        return new HttpGet(url);
    }

    @Override
    public Rtb rtb() {
        return Rtb.Uc;
    }

    @Override
    protected FeedbackResponse convert(FeedbackRequest request, String body) {
        return JsonHelper.convert(body, Response.class);
    }


    private static class Response extends FeedbackResponse {

        private String status;
        private String message;

        @Override
        public boolean isSuccess() {
            return StringUtils.equals("0", status);
        }

        public String getStatus() {
            return status;
        }

        public void setStatus(String status) {
            this.status = status;
        }

        public String getMessage() {
            return message;
        }

        public void setMessage(String message) {
            this.message = message;
        }
    }
}
