package com.bxm.adscounter.rtb.common.impl.weibo;

import com.alibaba.fastjson.JSON;
import com.bxm.adscounter.rtb.common.FailType;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.feedback.ActionType;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.impl.AbstractClickTrackerRtbIntegration;
import com.bxm.adsmanager.facade.model.adTag.TblAdTagVo;
import com.bxm.adsmanager.facade.service.AdTagFacadeService;
import com.bxm.adsprod.facade.tag.Tag;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.adsprod.facade.ticket.TicketKeyGenerator;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.openlog.sdk.consts.Inads;
import com.bxm.openlog.sdk.params.ProductionCommonParam;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.utils.UrlHelper;
import com.bxm.warcar.utils.http.OkHttpUtils;
import com.bxm.warcar.xcache.Fetcher;
import com.bxm.warcar.xcache.TargetFactory;
import com.google.common.collect.Maps;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpHeaders;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

/**
 * @author daill
 * @date 2022-06-30
 * @since 1.0
 */
@Slf4j
public class WeiBoRtbIntegration extends AbstractClickTrackerRtbIntegration {

    public static final String CLICK_ID = "mark_id";
    private final WeiBoConfig config;
    private Fetcher fetcher;
    @Autowired(required = false)
    private AdTagFacadeService tagFacadeService;

    public WeiBoRtbIntegration(WeiBoConfig config) {
        super(config);
        this.config = config;
    }

    private static CloseableHttpClient createHttpClient() {
        RequestConfig requestConfig = RequestConfig.custom().setSocketTimeout(4000)
                .setConnectionRequestTimeout(1000)
                .setConnectTimeout(1000).build();
        return HttpClientBuilder.create().setDefaultRequestConfig(requestConfig).build();
    }

    @Autowired
    public void setFetcher(Fetcher fetcher) {
        this.fetcher = fetcher;
    }

    @Override
    public String getClickId(KeyValueMap clickTrackerKeyValueMap) {
        return clickTrackerKeyValueMap.getFirst(Common.Param.CLICK_ID);
    }

    @Override
    public String getClickIdOnInadsAdClickLog(KeyValueMap clickEventLog) {
        return UrlHelper.getFirstValueOfParamName(clickEventLog.getRef(), CLICK_ID);
    }

    @Override
    public String getAdGroupId(FeedbackRequest request) {
        KeyValueMap keyValueMap = request.getKeyValueMap();
        if (Objects.isNull(keyValueMap)) {
            return null;
        }
        ActionType actionType = request.getActionType();
        if (Objects.isNull(actionType)) {
            return keyValueMap.getFirst(Inads.Param.TAGID);
        }
        PositionRtb positionRtb = request.getConfig();
        if (positionRtb.isFeedbackModeSummaryCpa()) {
            return keyValueMap.getFirst(Inads.Param.TAGID);
        }
        switch (actionType) {
            case TICKET_CLICK:
            case TICKET_CONVERSION:
                return keyValueMap.getFirst(Inads.Param.ADID);
            default:
                return keyValueMap.getFirst(Inads.Param.TAGID);
        }
    }

    @Override
    protected HttpRequestBase create(FeedbackRequest request) throws RtbIntegrationException {
        String url = getFeedbackUrl();
        String referrer = request.getReferrer();
        String eventType = request.getEventType();
        PositionRtb positionRtb = request.getConfig();
        String markId = UriComponentsBuilder.fromUriString(referrer).build().getQueryParams().getFirst(CLICK_ID);
        if (StringUtils.isBlank(markId)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, positionRtb.getPositionId() + "：Cannot found 'mark_id' by referrer.");
        }
        String appId = request.getConfig().getCustomerId();
        if (StringUtils.isBlank(appId)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, positionRtb.getPositionId() + "：Cannot found 'appId' by weibo.");
        }
        WBToken wbToken = fetchWeiboTokenByAppId(appId);
        if (Objects.isNull(wbToken) || StringUtils.isEmpty(wbToken.getAccess_token())) {
            throw new RtbIntegrationException(FailType.IllegalParameter, appId + "Cannot found 'wbToken' by weibo.");
        }
        Map<String, Object> params = Maps.newHashMap();
        params.put("time", System.currentTimeMillis());
        params.put("mark_id", UrlHelper.urlDecode(markId));
        params.put("behavior", eventType);
        if (Objects.equals(BehaviorEnum.ORDER.code, eventType)) {
            params.put("item_order_pay", UrlHelper.urlEncode(buildOrderInfo(request)));
        }
        url = OkHttpUtils.appendParams(url, params);
        HttpGet httpGet = new HttpGet(url);
        httpGet.addHeader(HttpHeaders.AUTHORIZATION, "Bearer " + wbToken.getAccess_token());
        httpGet.addHeader(HttpHeaders.ACCEPT, "application/json,application/text+gw2.0");
        return httpGet;
    }

    private String buildOrderInfo(FeedbackRequest request) {
        String time = String.valueOf(System.currentTimeMillis());
        PositionRtb positionRtb = request.getConfig();
        KeyValueMap map = request.getKeyValueMap();
        Ticket ticket = fetcher.hfetch(new TargetFactory<Ticket>()
                .keyGenerator(TicketKeyGenerator.getAllTickets())
                .field(map.getFirst(Inads.Param.ADID))
                .skipNativeCache(true)
                .cls(Ticket.class)
                .build());
        //计算复杂参数
        String goodsName = map.getFirst(Inads.Param.GOODS_NAME);
        if (StringUtils.isEmpty(goodsName) && CollectionUtils.isNotEmpty(ticket.getTags())) {
            List<Tag> tags = ticket.getTags();
            TblAdTagVo tag = tagFacadeService.findAppointLevelTag(Integer.parseInt(tags.get(tags.size() - 1).getCode()), 2, 2).getReturnValue();
            goodsName = tag != null ? tag.getName() : goodsName;
        }
        Integer cpaPrice = request.getConversionLevel() == FeedbackRequest.DEEP_CONVERSION_LEVEL ? ticket.getDeepCpaPrice() : ticket.getCpaPrice();
        //构建对象
        OrderInfo orderInfo = OrderInfo.builder().click_time(Optional.ofNullable(map.getFirst(ProductionCommonParam.TIME)).filter(StringUtils::isNotEmpty).orElse(time))
                .order_create_time(Optional.ofNullable(map.getFirst(Inads.Param.ORDER_CREATE_TIME)).filter(StringUtils::isNotEmpty).orElse(time))
                .order_pay_time(Optional.ofNullable(map.getFirst(Inads.Param.ORDER_PAY_TIME)).filter(StringUtils::isNotEmpty).orElse(time))
                .order_pay_click_time(Optional.ofNullable(map.getFirst(Inads.Param.ORDER_PAY_TIME)).filter(StringUtils::isNotEmpty).orElse(time))
                .item_name(goodsName)
                .item_price(Optional.ofNullable(map.getFirst(Inads.Param.GOODS_PRICE)).filter(StringUtils::isNotEmpty).map(Float::parseFloat).orElse(cpaPrice / 1000f))
                .payment_amount(Optional.ofNullable(map.getFirst(Inads.Param.PAY_AMOUNT)).filter(StringUtils::isNotEmpty).map(Float::parseFloat).orElse(cpaPrice / 1000f))
                .quantity_of_item(Optional.ofNullable(map.getFirst(Inads.Param.GOODS_QUANTITY)).filter(StringUtils::isNotEmpty).map(Integer::parseInt).orElse(1))
                .category(request.getConversionLevel() == FeedbackRequest.SHALLOW_CONVERSION_LEVEL ? positionRtb.getCidCategoryOne() : positionRtb.getCidCategoryTwo())
                .order_status("已付款")
                .order_id(Optional.ofNullable(map.getFirst(Inads.Param.ORDER_ID)).filter(StringUtils::isNotEmpty).orElse(map.getFirst(Inads.Param.BXMID)))
                .service_provider("变现猫")
                .build();
        return JSON.toJSONString(orderInfo);
    }

    private WBToken fetchWeiboTokenByAppId(String appId) {
        return fetcher.hfetch(new TargetFactory<WBToken>()
                .keyGenerator(() -> KeyBuilder.build("wb", "token"))
                .field(appId)
                .skipNativeCache(true)
                .cls(WBToken.class)
                .build());
    }

    @Override
    public Rtb rtb() {
        return Rtb.WeiBo;
    }

    @Override
    protected FeedbackResponse convert(FeedbackRequest request, String body) {
        return JsonHelper.convert(body, Response.class);
    }

    private static class Response extends FeedbackResponse {

        /**
         * 重复回传
         */
        private static final String CODE_UNIQ_REPEAT = "40001";

        /**
         * AccessToken 已过期
         */
        private static final String CODE_AK_EXPIRED = "40113";

        private String code;
        private String message;

        @Override
        public boolean isSuccess() {
            return StringUtils.equals("0", code) || StringUtils.equals(CODE_UNIQ_REPEAT, code);
        }

        public String getCode() {
            return code;
        }

        public void setCode(String code) {
            this.code = code;
        }

        public String getMessage() {
            return message;
        }

        public void setMessage(String message) {
            this.message = message;
        }
    }

    @Data
    static class WBToken {
        private String access_token;
        private String expires_in;
        private String exDate;
    }


    enum BehaviorEnum {
        ORDER("2001"),
        PROGRAM("3003");
        private final String code;

        BehaviorEnum(String code) {
            this.code = code;
        }
    }

    @Data
    @AllArgsConstructor
    @NoArgsConstructor
    @Builder
    static class OrderInfo {
        private String click_time;
        private String order_create_time;
        private String order_pay_time;
        private String order_pay_click_time;
        private String item_name;
        private Float item_price;
        private Float payment_amount;
        private Integer quantity_of_item;
        private String category;
        private String order_status;
        private String order_id;
        private String service_provider;
        private Integer order_type;
        private Integer is_same_item;
    }
}
