package com.bxm.adscounter.rtb.common.impl.xiaomi;

import com.bxm.adscounter.rtb.common.FailType;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.impl.AbstractClickTrackerRtbIntegration;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.warcar.utils.UrlHelper;
import com.bxm.warcar.utils.http.OkHttpUtils;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;

/**
 * @author tangxiao
 * @date 2022-06-01
 * @since 1.0
 */
@Slf4j
public class XiaomiRtbIntegration extends AbstractClickTrackerRtbIntegration {

    public static final String CLICK_ID = "logExtra";

    public XiaomiRtbIntegration(XiaomiConfig config) {
        super(config, createHttpClient());
    }

    @Override
    public Rtb rtb() {
        return Rtb.XiaoMi;
    }

    @Override
    protected boolean isReadBodyForHttpResponse() {
        return false;
    }

    @Override
    public String getClickId(KeyValueMap clickTrackerKeyValueMap) {
        return clickTrackerKeyValueMap.getFirst(Common.Param.CLICK_ID);
    }

    @Override
    public String getClickIdOnInadsAdClickLog(KeyValueMap clickEventLog) {
        return UrlHelper.getFirstValueOfParamName(clickEventLog.getRef(), CLICK_ID);
    }

    @Override
    public String getAdGroupId(FeedbackRequest request) {
        String referrer = request.getReferrer();
        UriComponents build = UriComponentsBuilder.fromUriString(referrer).build();
        MultiValueMap<String, String> queryParams = build.getQueryParams();
        String clickId = queryParams.getFirst(CLICK_ID);
        KeyValueMap clickTracker = getClickTracker(clickId);

        return Optional.ofNullable(clickTracker)
                .map(k -> k.getFirst(Common.Param.AD_GROUP_ID))
                .orElse(StringUtils.EMPTY);
    }

    @Override
    protected HttpRequestBase create(FeedbackRequest request) throws RtbIntegrationException {
        String url = getFeedbackUrl();
        String referrer = request.getReferrer();
        String eventType = request.getEventType();
        MultiValueMap<String, String> queryParams = UriComponentsBuilder.fromUriString(referrer).build().getQueryParams();
        String conversionId = queryParams.getFirst("conversionId");
        String logExtra = queryParams.getFirst("logExtra");
        String webConversionId = queryParams.getFirst("webConversionId");
        String conversion_debug = queryParams.getFirst("conversion_debug");
        if (StringUtils.isBlank(conversionId)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'conversionId' by referrer.");
        }
        if (StringUtils.isBlank(logExtra)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'logExtra' by referrer.");
        }
        if (StringUtils.isBlank(webConversionId)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'webConversionId' by referrer.");
        }
        Map<String, Object> params = Maps.newHashMap();
        params.put("conversionId", conversionId);
        params.put("eventType", "form");
        params.put("clientTime", String.valueOf(System.currentTimeMillis()));
        params.put("logExtra", logExtra);
        params.put("webConversionId", webConversionId);
        params.put("convType", eventType);

        if (StringUtils.isNotBlank(conversion_debug) && Objects.equals(conversion_debug, "true")) {
            params.put("conversion_debug", conversion_debug);
        }

        url = OkHttpUtils.appendParams(url, params);
        return new HttpGet(url);
    }

    @Override
    protected FeedbackResponse convert(FeedbackRequest request, String body) throws RtbIntegrationException {
        FeedbackResponse feedbackResponse = new FeedbackResponse();
        feedbackResponse.setSuccess(true);
        return feedbackResponse;
    }


    private static CloseableHttpClient createHttpClient() {
        RequestConfig requestConfig = RequestConfig.custom().setSocketTimeout(2000).setConnectionRequestTimeout(1000).setConnectTimeout(1000).build();
        return HttpClientBuilder.create().setDefaultRequestConfig(requestConfig).build();
    }

}
