package com.bxm.adscounter.rtb.common.impl.xmly;

import com.bxm.adscounter.rtb.common.FailType;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.impl.AbstractClickTrackerRtbIntegration;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.UrlHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.Optional;

/**
 * @author daill
 * @date 2022-05-16
 * @since 1.0
 */
@Slf4j
public class HimalayanRtbIntegration extends AbstractClickTrackerRtbIntegration {

    public static final String CLICK_ID = "callback";
    public static final String AD_GROUP_ID = "materialid";

    public HimalayanRtbIntegration(HimalayanConfig config) {
        super(config);
    }

    @Override
    public String getClickId(KeyValueMap clickTrackerKeyValueMap) {
        return clickTrackerKeyValueMap.getFirst(Common.Param.CLICK_ID);
    }

    @Override
    public String getClickIdOnInadsAdClickLog(KeyValueMap clickEventLog) {
        String ref = clickEventLog.getRef();
        String clickId = UrlHelper.getFirstValueOfParamName(ref, CLICK_ID);
        if (StringUtils.isBlank(clickId)) {
            log.warn("Not found: {}", ref);
            return null;
        }
        return DigestUtils.md5Hex(UrlHelper.urlDecode(clickId));
    }

    @Override
    public String getAdGroupId(FeedbackRequest request) {
        String referrer = request.getReferrer();
        UriComponents build = UriComponentsBuilder.fromUriString(referrer).build();
        MultiValueMap<String, String> queryParams = build.getQueryParams();
        String clickId = queryParams.getFirst(CLICK_ID);
        if (StringUtils.isBlank(clickId)) {
            log.warn("Not found: {}", referrer);
            return null;
        }
        KeyValueMap clickTracker = getClickTracker(DigestUtils.md5Hex(UrlHelper.urlDecode(clickId)));

        return Optional.ofNullable(clickTracker)
                .map(k -> k.getFirst(Common.Param.AD_GROUP_ID))
                .orElse(StringUtils.EMPTY);
    }

    @Override
    protected HttpRequestBase create(FeedbackRequest request) throws RtbIntegrationException {
        String referrer = request.getReferrer();
        String eventType = request.getEventType();

        String callback = UriComponentsBuilder.fromUriString(referrer).build().getQueryParams().getFirst("callback");
        if (StringUtils.isBlank(callback)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'callback' by referrer.");
        }
        callback = UrlHelper.urlDecode(callback);
        return new HttpGet(callback.trim() + "&event_type=" + eventType);
    }

    @Override
    public Rtb rtb() {
        return Rtb.Ximalaya;
    }

    @Override
    protected FeedbackResponse convert(FeedbackRequest request, String body) {
        return JsonHelper.convert(body, Response.class);
    }

    private static class Response extends FeedbackResponse {
        private String type;
        private String text;
        private String data;
        private String errors;

        @Override
        public boolean isSuccess() {
            return StringUtils.equalsIgnoreCase("SUCCESS", type);
        }

        public String getType() {
            return type;
        }

        public void setType(String type) {
            this.type = type;
        }

        public String getText() {
            return text;
        }

        public void setText(String text) {
            this.text = text;
        }

        public String getData() {
            return data;
        }

        public void setData(String data) {
            this.data = data;
        }

        public String getErrors() {
            return errors;
        }

        public void setErrors(String errors) {
            this.errors = errors;
        }
    }
}
